import React, {useMemo} from 'react';
import {useSelector} from 'react-redux';

import {EProjectName} from 'constants/common';

import {IWithClassName} from 'types/withClassName';
import EHeaderBorderBottomType from './types/EHeaderBorderBottomType';

import experimentsSelector from 'selectors/common/experimentsSelector';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import HeaderMobile from './components/HeaderMobile/HeaderMobile';
import HeaderDesktop from './components/HeaderDesktop/HeaderDesktop';

interface IHeaderProps extends IWithClassName, IWithQaAttributes {
    containerClassName?: string;

    wrapperClassName?: string;

    formWrapperClassName?: string;

    searchFormCollapsedClassName?: string;
    /**
     * Шапка будет ездить вместе со скроллом.
     */
    isFixed?: boolean;

    /**
     * Показывать навигацию.
     */
    showNavigation?: boolean;

    /**
     * Показывать поисковую форму.
     */
    showSearchForm?: boolean;

    /**
     * Шапка зафиксирована в одном состоянии, котрое определяется searchFormInitialIsExpanded
     */
    searchFormIsStatic?: boolean;

    /**
     * Признак изначального состояния календаря
     * Имеет смысл для статичного оотображения на десктопе
     */
    initiallyCalendarIsOpen?: boolean;

    /**
     * Признак валидации на маунт компонента
     * Необходимо для подсветки ошибок формы при загрузке страницы
     */
    validateSearchFormOnMount?: boolean;

    /**
     * По-умолчанию форма развернута/свернута.
     */
    searchFormInitialIsExpanded?: boolean;

    /**
     * Активная вертикаль. Влияет на то, какую поисковую форму нужно показывать
     * если задан `showSearchForm` и влияет на формирование ссылки для лого Путешествий
     */
    project?: EProjectName;

    /**
     * Показывать ли навигацию по проектам в выезжающей шторке
     */
    hasSideSheetNavigation?: boolean;

    /**
     * Отображение компактных фильтров справа от строки поиска
     */
    filters?: React.ReactNode;

    /**
     * Скрывает в тачах блок с логотипом и пользователем
     */
    hideTopMobileBlock?: boolean;

    hideSearchInformation?: boolean;

    borderBottomType?: EHeaderBorderBottomType;

    withYandexLogo?: boolean;
    withNavCaptions?: boolean;

    onSubmitForm?(): void;
}

export interface IHeaderDeviceBase extends IWithClassName, IWithQaAttributes {
    isFixed?: boolean;
    showNavigation?: boolean;
    showSearchForm?: boolean;
    project?: EProjectName;
    containerClassName?: string;
    wrapperClassName?: string;
    formWrapperClassName?: string;
    searchFormIsAvailable: boolean;
    validateSearchFormOnMount?: boolean;
    onSubmitForm?(): void;
    withNavCaptions?: boolean;
}

const ROOT_QA = 'portal-header';

const Header: React.FC<IHeaderProps> = props => {
    const {
        className,
        containerClassName,
        wrapperClassName,
        formWrapperClassName,
        searchFormCollapsedClassName,
        isFixed,
        showNavigation,
        showSearchForm,
        searchFormIsStatic,
        initiallyCalendarIsOpen,
        validateSearchFormOnMount,
        searchFormInitialIsExpanded,
        hasSideSheetNavigation,
        project,
        filters,
        onSubmitForm,
        hideTopMobileBlock,
        hideSearchInformation,
        borderBottomType = EHeaderBorderBottomType.FULL,
        withYandexLogo = true,
        withNavCaptions,
    } = props;

    const deviceType = useDeviceType();
    const experiments = useSelector(experimentsSelector);

    const searchFormIsAvailable = useMemo(() => {
        return Boolean(
            showSearchForm &&
                project &&
                [
                    EProjectName.TRAINS,
                    EProjectName.AVIA,
                    EProjectName.HOTELS,
                    EProjectName.BUSES,
                ].includes(project),
        );
    }, [project, showSearchForm]);

    if (deviceType.isMobile) {
        return (
            <HeaderMobile
                className={className}
                isFixed={isFixed}
                hasSideSheetNavigation={hasSideSheetNavigation}
                showNavigation={showNavigation}
                validateSearchFormOnMount={validateSearchFormOnMount}
                searchFormInitialIsExpanded={searchFormInitialIsExpanded}
                showSearchForm={showSearchForm}
                project={project}
                searchFormIsAvailable={searchFormIsAvailable}
                filters={filters}
                onSubmitForm={onSubmitForm}
                hideTopMobileBlock={hideTopMobileBlock}
                searchFormCollapsedClassName={searchFormCollapsedClassName}
                hideSearchInformation={hideSearchInformation}
                borderBottomType={borderBottomType}
                withNavCaptions={withNavCaptions}
                {...prepareQaAttributes(ROOT_QA)}
            />
        );
    }

    return (
        <HeaderDesktop
            className={className}
            containerClassName={containerClassName}
            wrapperClassName={wrapperClassName}
            formWrapperClassName={formWrapperClassName}
            isFixed={isFixed}
            showNavigation={showNavigation}
            showSearchForm={showSearchForm}
            searchFormIsStatic={searchFormIsStatic}
            initiallyCalendarIsOpen={initiallyCalendarIsOpen}
            validateSearchFormOnMount={validateSearchFormOnMount}
            searchFormInitialIsExpanded={searchFormInitialIsExpanded}
            project={project}
            deviceType={deviceType}
            searchFormIsAvailable={searchFormIsAvailable}
            onSubmitForm={onSubmitForm}
            experiments={experiments}
            hideSearchInformation={hideSearchInformation}
            borderBottomType={borderBottomType}
            withYandexLogo={withYandexLogo}
            withNavCaptions={withNavCaptions}
            {...prepareQaAttributes(ROOT_QA)}
        />
    );
};

export default React.memo(Header);
