import React, {useCallback, useMemo} from 'react';

import {EProjectName} from 'constants/common';

import {ECommonGoal} from 'utilities/metrika/types/goals/common';
import EHeaderBorderBottomType from 'components/Header/types/EHeaderBorderBottomType';

import {loadable} from 'utilities/componentLoadable';
import {reachGoal} from 'utilities/metrika';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {getQa, prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import ModalWithHistoryBack from 'containers/withSupportHistoryBack/ModalWithHistoryBack/ModalWithHistoryBack';

import Modal from 'components/Modal/Modal';
import {IHeaderDeviceBase} from 'components/Header/Header';
import YandexTravelLogo from 'components/YandexTravelLogo/YandexTravelLogo';
import NavigationContainer from 'components/NavigationContainer/NavigationContainer';
import UserInfo from '../../components/UserInfo/UserInfo';

import cx from './HeaderMobile.scss';

const FORM_LOADABLE_FALLBACK_OPTIONS = {
    hasFallback: true,
    className: cx('formChunkLoader'),
};

/* SearchForm */
const HotelsSearchForm = loadable(
    () => import('projects/depreacted/hotels/components/SearchForm/SearchForm'),
    FORM_LOADABLE_FALLBACK_OPTIONS,
);

/* SearchInformation */
const HotelsSearchInformation = loadable(
    () =>
        import(
            'projects/depreacted/hotels/containers/HotelsSearchInformation/HotelsSearchInformation'
        ),
);

/* NavigationSideSheet */
const NavigationSideSheet = loadable(
    () => import('components/NavigationSideSheet/NavigationSideSheet'),
);

interface IHeaderMobileProps extends IHeaderDeviceBase {
    filters?: React.ReactNode;
    hasSideSheetNavigation?: boolean;
    searchFormInitialIsExpanded?: boolean;
    hideTopMobileBlock?: boolean;
    searchFormCollapsedClassName?: string;
    hideSearchInformation?: boolean;
    borderBottomType: EHeaderBorderBottomType;
}

const SUPPORTED_PROJECTS = [
    EProjectName.AVIA,
    EProjectName.TRAINS,
    EProjectName.HOTELS,
    EProjectName.BUSES,
];

const HeaderMobile: React.FC<IHeaderMobileProps> = props => {
    const {
        className,
        searchFormCollapsedClassName,
        project,
        showNavigation,
        hasSideSheetNavigation,
        searchFormIsAvailable,
        validateSearchFormOnMount,
        filters,
        onSubmitForm,
        hideTopMobileBlock,
        hideSearchInformation,
        borderBottomType,
        withNavCaptions,
    } = props;

    const deviceType = useDeviceType();

    const {
        value: modalIsVisible,
        setTrue: showModal,
        setFalse: hideModal,
    } = useBoolean(false);

    const portalLogoQa = getQa({
        parent: props,
        current: 'portalLogo',
    });

    const userInfoQa = getQa({
        parent: props,
        current: 'userInfo',
    });

    const navigationSideSheetQa = getQa({
        parent: props,
        current: 'navigationSideSheet',
    });

    const onCollapsedSearchFormClick = useCallback(() => {
        showModal();

        reachGoal(ECommonGoal.HEADER_SEARCH_FORM_INFORMATION_CLICK);
    }, [showModal]);

    const handleSubmitForm = useCallback(() => {
        onSubmitForm?.();

        hideModal();
    }, [hideModal, onSubmitForm]);

    const searchInformationBlock = useMemo(() => {
        if (hideSearchInformation) return null;

        switch (project) {
            case EProjectName.HOTELS: {
                return (
                    <HotelsSearchInformation
                        className={cx(
                            'searchFormCollapsed',
                            searchFormCollapsedClassName,
                        )}
                        onClick={onCollapsedSearchFormClick}
                        deviceType={deviceType}
                    />
                );
            }

            default:
                return null;
        }
    }, [
        deviceType,
        onCollapsedSearchFormClick,
        project,
        filters,
        searchFormCollapsedClassName,
        hideSearchInformation,
    ]);

    const searchFormBlock = useMemo(() => {
        switch (project) {
            case EProjectName.HOTELS: {
                return (
                    <HotelsSearchForm
                        className={cx('searchForm')}
                        onSubmit={handleSubmitForm}
                    />
                );
            }

            default:
                return null;
        }
    }, [handleSubmitForm, project, validateSearchFormOnMount]);

    const topBlock = useMemo(() => {
        if (hideTopMobileBlock) {
            return null;
        }

        return (
            <div className={cx('wrapper', 'wrapper_mobile')}>
                <div className={cx('main')}>
                    <YandexTravelLogo
                        className={cx('logo')}
                        isSmall
                        project={project}
                        {...prepareQaAttributes(portalLogoQa)}
                    />

                    <UserInfo
                        hasSideSheetNavigation={hasSideSheetNavigation}
                        withNavCaptions={withNavCaptions}
                        {...prepareQaAttributes(userInfoQa)}
                    />

                    <NavigationSideSheet
                        className={cx('navigationSideSheet')}
                        {...prepareQaAttributes(navigationSideSheetQa)}
                    />
                </div>
            </div>
        );
    }, [
        hasSideSheetNavigation,
        hideTopMobileBlock,
        navigationSideSheetQa,
        portalLogoQa,
        project,
        userInfoQa,
        withNavCaptions,
    ]);

    return (
        <header
            className={cx(
                'root',
                className,
                `root_project_${project}`,
                `root_border_${borderBottomType}`,
            )}
            {...prepareQaAttributes(props)}
        >
            {topBlock}

            {showNavigation && !hasSideSheetNavigation && (
                <div className={cx('navigationWrapper')}>
                    <div className={cx('navigationScroller')}>
                        <NavigationContainer
                            className={cx('navigation')}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'navigation',
                            })}
                        />
                    </div>
                </div>
            )}
            {searchFormIsAvailable &&
                project &&
                SUPPORTED_PROJECTS.includes(project) && (
                    <>
                        <div>
                            <div className={cx('wrapper', 'wrapper_mobile')}>
                                {searchInformationBlock}
                            </div>
                        </div>

                        <ModalWithHistoryBack
                            isMobile
                            fullScreen
                            disableAutoFocus
                            isVisible={modalIsVisible}
                            verticalAlign="top"
                            autoclosable={false}
                            disableAnimations
                            onClose={hideModal}
                        >
                            <Modal.Content>{searchFormBlock}</Modal.Content>
                        </ModalWithHistoryBack>
                    </>
                )}
        </header>
    );
};

export default HeaderMobile;
