import React, {useCallback, useMemo} from 'react';
import {useLocation, matchPath} from 'react-router-dom';
import {useDispatch, useSelector} from 'react-redux';

import {URLs} from 'constants/urls';

import {ECommonGoal} from 'utilities/metrika/types/goals/common';
import {IWithClassName} from 'types/withClassName';
import {EUserAvatarSize} from 'components/User/types';
import {EFavoritesGoal} from 'utilities/metrika/types/goals/favorites';

import {getUserUndeadMailCount} from 'selectors/common/getUserUndeadMailCount';

import {internalUrl} from 'utilities/url';
import {reachGoal} from 'utilities/metrika';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useSingleProjectView} from 'utilities/hooks/useSingleProjectView';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';
import {useUniqSvg} from 'utilities/hooks/useUniqSvg';
import {useUserInfo} from 'utilities/hooks/useUserInfo';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nBlock from 'i18nNew/index';

import User from 'components/User/User';
import Link from 'components/Link/Link';
import Flex from 'components/Flex/Flex';
import YandexMessenger from 'components/YandexMessenger/YandexMessenger';
import PlusHeaderIcon from './components/PlusHeaderIcon/PlusHeaderIcon';
import CertificateLink from './components/CertificateLink/CertificateLink';
import OrdersIcon from 'icons/24/Orders';
import HeartFilledIcon from 'icons/24/HeartFilled';
import OrdersFilledIcon from 'icons/24/OrdersFilled';
import HeartOutlineIcon from 'icons/24/HeartOutline';
import Text from 'components/Text/Text';
import TripsLink from 'components/Header/components/UserInfo/components/TripsLink/TripsLink';

import cx from './UserInfo.scss';

const accountClickHandler = (): void => reachGoal(ECommonGoal.ACCOUNT_TAB);

interface IUserInfoProps extends IWithClassName, IWithQaAttributes {
    hasSideSheetNavigation?: boolean;
    withNavCaptions?: boolean;
}

const UserInfo: React.FC<IUserInfoProps> = ({
    className,
    hasSideSheetNavigation,
    withNavCaptions,
    ...rest
}) => {
    const dispatch = useDispatch();
    const deviceType = useDeviceType();
    const userInfo = useUserInfo();
    const isSingleProjectView = useSingleProjectView();
    const unreadMailCount = useSelector(getUserUndeadMailCount);
    const fetchUserUnreadMailCount = useCallback(() => {
        // dispatch(mailActions.fetchUserUnreadMailCount());
    }, [dispatch]);
    const location = useLocation();
    const ref = useUniqSvg<HTMLDivElement>();

    const {isMobile, isDesktop} = deviceType;

    const showTickets =
        isDesktop && !hasSideSheetNavigation && !isSingleProjectView;
    const isAuth = isAuthUser(userInfo);

    const showUser = isDesktop && !(hasSideSheetNavigation && isAuth);

    const isOnAccountPage = useMemo(() => {
        return Boolean(
            matchPath(location.pathname, {
                path: URLs.account,
                exact: false,
            }),
        );
    }, [location.pathname]);

    const isOnFavoritesPage = useMemo(() => {
        return Boolean(
            matchPath(location.pathname, {
                path: URLs.favorites,
                exact: true,
            }),
        );
    }, [location.pathname]);

    const isOnHotelsHappyPage = useMemo(() => {
        return Boolean(
            matchPath(location.pathname, {
                path: URLs.hotelsBookSuccess,
                exact: true,
            }),
        );
    }, [location.pathname]);

    const isOnHappyPage = useMemo(() => {
        return (
            isOnHotelsHappyPage ||
            [URLs.aviaBookingResult, URLs.trainsOrderHappyPage].some(path =>
                matchPath(location.pathname, {
                    path,
                    exact: true,
                }),
            )
        );
    }, [isOnHotelsHappyPage, location.pathname]);

    const onFavoritesClick = useCallback(() => {
        reachGoal(EFavoritesGoal.HEADER_FAVORITES_CLICK);
    }, []);

    return (
        <div
            className={cx('root', deviceMods('root', deviceType), className)}
            ref={ref}
            {...prepareQaAttributes(rest)}
        >
            <Flex
                className={cx('icons')}
                between={isMobile || !withNavCaptions ? 4 : 5}
                inline
                alignItems="center"
            >
                {isMobile && (
                    <PlusHeaderIcon
                        className={cx('plus')}
                        {...prepareQaAttributes({
                            parent: rest,
                            current: 'plus',
                        })}
                    />
                )}

                {isDesktop && (
                    <CertificateLink
                        title={i18nBlock.navCertificates()}
                        label={
                            withNavCaptions
                                ? i18nBlock.navCertificates()
                                : undefined
                        }
                    />
                )}

                {isDesktop && (
                    <YandexMessenger
                        entrypoint="header"
                        metrikaGoal={ECommonGoal.PORTAL_CHAT_HEADER_CLICK}
                        title={i18nBlock.navSupport()}
                        label={
                            withNavCaptions ? i18nBlock.navSupport() : undefined
                        }
                        labelSize="s"
                    />
                )}

                <Link
                    to={internalUrl(URLs.favorites)}
                    theme="black"
                    title={i18nBlock.navFavorites()}
                    onClick={onFavoritesClick}
                    {...prepareQaAttributes({
                        parent: rest,
                        current: 'favoriteLink',
                    })}
                >
                    <Flex inline alignItems="center" between={2}>
                        {isOnFavoritesPage ? (
                            <HeartFilledIcon />
                        ) : (
                            <HeartOutlineIcon />
                        )}

                        {withNavCaptions && (
                            <Text
                                className={cx('text')}
                                color="current"
                                size="s"
                            >
                                {i18nBlock.navFavorites()}
                            </Text>
                        )}
                    </Flex>
                </Link>

                {showTickets && (
                    <TripsLink
                        to={internalUrl(URLs.accountTrips)}
                        title={i18nBlock.navTrips()}
                        isOnHappyPage={isOnHappyPage}
                        isOnHotelsHappyPage={isOnHotelsHappyPage}
                        onClick={accountClickHandler}
                        {...prepareQaAttributes({
                            parent: rest,
                            current: 'accountLink',
                        })}
                    >
                        {isOnAccountPage ? (
                            <OrdersFilledIcon />
                        ) : (
                            <OrdersIcon />
                        )}

                        {withNavCaptions && (
                            <Text
                                className={cx('text')}
                                color="current"
                                size="s"
                            >
                                {i18nBlock.navTrips()}
                            </Text>
                        )}
                    </TripsLink>
                )}
            </Flex>

            {isDesktop && (
                <PlusHeaderIcon
                    className={cx('plus')}
                    {...prepareQaAttributes({
                        parent: rest,
                        current: 'plus',
                    })}
                />
            )}

            {showUser && (
                <div
                    {...prepareQaAttributes({
                        parent: rest,
                        current: 'login',
                    })}
                >
                    <User
                        className={cx('user')}
                        userInfo={userInfo}
                        avatarSize={
                            isMobile ? EUserAvatarSize.S : EUserAvatarSize.L
                        }
                        unreadMailCount={unreadMailCount}
                        fetchUserUnreadMailCount={fetchUserUnreadMailCount}
                    />
                </div>
            )}
        </div>
    );
};

export default UserInfo;
