import {FC, MouseEventHandler, ReactNode, memo, useEffect, useRef} from 'react';
import {useSelector} from 'react-redux';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useBoolean} from 'utilities/hooks/useBoolean';

import * as i18nBlock from 'i18nNew/trips-common';

import Flex from 'components/Flex/Flex';
import Link from 'components/Link/Link';
import Text from 'components/Text/Text';
import MessageBoxPopup, {
    EMessageBoxPopupTheme,
} from 'components/MessageBoxPopup/MessageBoxPopup';
import NotificationDot from 'components/NotificationDot/NotificationDot';

import cx from './TripsLink.scss';

interface ITripsLinkProps extends IWithQaAttributes {
    to: string;
    title: string;
    isOnHappyPage: boolean;
    isOnHotelsHappyPage: boolean;
    children?: ReactNode;
    onClick?: MouseEventHandler;
}

const TripsLink: FC<ITripsLinkProps> = props => {
    const {to, title, isOnHappyPage, isOnHotelsHappyPage, children, onClick} =
        props;

    const {
        value: isTooltipVisible,
        setTrue: openTooltip,
        setFalse: closeTooltip,
    } = useBoolean(false);
    const anchorRef = useRef<HTMLDivElement | null>(null);
    const {isDesktop} = useDeviceType();
    const isNewTrip = useSelector(isNewTripSelector);

    useEffect(() => {
        if (isOnHappyPage && isNewTrip) {
            openTooltip();
        }
    }, [isNewTrip, isOnHappyPage, openTooltip]);

    return (
        <>
            <Link
                to={to}
                theme="black"
                onClick={onClick}
                title={title}
                {...prepareQaAttributes(props)}
            >
                <NotificationDot size="m" dotVisible={isNewTrip}>
                    <Flex
                        inline
                        alignItems="center"
                        between={2}
                        innerRef={anchorRef}
                    >
                        {children}
                    </Flex>
                </NotificationDot>
            </Link>

            {isDesktop && (
                <MessageBoxPopup
                    theme={EMessageBoxPopupTheme.WHITE}
                    isVisible={isTooltipVisible}
                    anchorRef={anchorRef}
                    onClose={closeTooltip}
                >
                    <div className={cx('tooltipContent')}>
                        <Text size="m">
                            {isOnHotelsHappyPage
                                ? i18nBlock.newTripHotelDescription()
                                : i18nBlock.newTripTicketDescription()}
                        </Text>
                    </div>
                </MessageBoxPopup>
            )}
        </>
    );
};

export default memo(TripsLink);
