import React, {useEffect, useRef} from 'react';
import {subscribe, unsubscribe} from 'pubsub-js';

import {EPubSubEvent} from 'types/EPubSubEvent';

import scrollTo from 'utilities/dom/scrollTo';
import getScrollWidth from 'utilities/dom/getScrollWidth';

function attachStyle(
    element: HTMLElement,
    style: {css: string; className: string},
) {
    element.classList.add(style.className);

    const removeStyleNode = attachStylesheet(style.css);

    return () => {
        removeStyleNode();
        element.classList.remove(style.className);
    };
}

export const HideBodyVerticalScrollClassName =
    'document-HideBodyVerticalScroll';

function generateDocumentStyle(
    documentMargin: number,
    previousScrollY: number,
) {
    const className = HideBodyVerticalScrollClassName;
    const css = `\
:root {
    --scroll-gap: ${documentMargin}px;
}

.${className} {
  overflow: hidden !important;
  margin-right: ${documentMargin}px !important;
  height: 100%;
}

.${className} #app {
  position: fixed;
  top: -${previousScrollY}px;
  width: calc(100% - ${documentMargin}px);
}
`;

    return {className, css};
}

function attachStylesheet(sheet: string) {
    const style = document.createElement('style');

    style.setAttribute('type', 'text/css');

    // @ts-ignore IE specific api
    if (style.styleSheet) {
        // @ts-ignore IE specific api
        style.styleSheet.cssText = sheet;
    } else {
        style.appendChild(document.createTextNode(sheet));
    }

    const head = document.getElementsByTagName('head')[0];

    head.appendChild(style);

    return () => {
        if (head.contains(style)) {
            head.removeChild(style);
        }
    };
}

let modalsCounter = 0;
let stopHidingScroll: (() => void) | null = null;

const HideBodyVerticalScroll: React.FC = () => {
    const initialWindowYScroll = useRef<number>();

    useEffect(() => {
        initialWindowYScroll.current = window.pageYOffset;

        // Подписка на программное изменение скролла
        // Нужна чтобы после закрытия модала установить скролл на новое значение
        const programScrollSubscription = subscribe(
            EPubSubEvent.SCROLL_CHANGED,
            (_event: EPubSubEvent, scrollOptions: ScrollToOptions) => {
                initialWindowYScroll.current = scrollOptions.top;
            },
        );

        const {
            documentElement,
            documentElement: {clientHeight, scrollHeight},
        } = document;

        if (modalsCounter === 0 && clientHeight < scrollHeight) {
            const documentComputedStyle = getComputedStyle(documentElement);
            const scrollWidth = getScrollWidth();
            const documentMarginRight = parseFloat(
                documentComputedStyle.marginRight || '',
            );
            const documentStyle = generateDocumentStyle(
                documentMarginRight + scrollWidth,
                initialWindowYScroll.current,
            );

            stopHidingScroll = attachStyle(documentElement, documentStyle);
        }

        modalsCounter++;

        return (): void => {
            modalsCounter--;

            if (modalsCounter === 0) {
                if (stopHidingScroll) {
                    stopHidingScroll();
                }

                unsubscribe(programScrollSubscription);
                scrollTo({top: initialWindowYScroll.current, left: 0});
            }
        };
    }, []);

    return null;
};

export default HideBodyVerticalScroll;
