import React, {createRef, useEffect, useRef} from 'react';

import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import HorizontalHiddenScroll from 'components/HorizontalHiddenScroll/HorizontalHiddenScroll';
import {TBoxSizes} from 'components/Box/Box';

import cx from './HorizontalScroller.scss';

export interface IHorizontalScroller
    extends IWithClassName,
        IWithQaAttributes,
        React.HTMLAttributes<HTMLDivElement> {
    offset?: TBoxSizes;
    padderClassName?: string;
    scrollableClassName?: string;
    scrollerRef?: React.RefObject<HTMLDivElement>;

    /**
     * Ref ноды к которой нужно проскролить при маунте компонента.
     */
    initialScrollToRef?: React.RefObject<HTMLElement>;
}

const HorizontalScroller: React.FC<IHorizontalScroller> = props => {
    const {
        className,
        padderClassName,
        children,
        offset,
        scrollableClassName,
        scrollerRef = createRef(),
        initialScrollToRef,
        ...rest
    } = props;
    const haveScrolledInitiallyRef = useRef(false);

    useEffect(() => {
        if (
            !scrollerRef?.current ||
            !initialScrollToRef?.current ||
            haveScrolledInitiallyRef.current
        ) {
            return;
        }

        const padder = scrollerRef.current.firstChild;

        if (!padder || !(padder instanceof Element)) {
            return;
        }

        const padderLeftPadding = parseInt(
            window.getComputedStyle(padder).paddingLeft,
            10,
        );

        scrollerRef.current.scrollLeft =
            initialScrollToRef.current.offsetLeft - padderLeftPadding;

        haveScrolledInitiallyRef.current = true;
    }, [initialScrollToRef, scrollerRef]);

    return (
        <HorizontalHiddenScroll
            className={cx(
                typeof offset === 'undefined'
                    ? 'root_offset_container'
                    : `root_offset_${offset}`,
                className,
            )}
            scrollableClassName={cx('scrollable', scrollableClassName)}
            scrollerRef={scrollerRef}
            {...rest}
            {...prepareQaAttributes(props)}
        >
            <div className={cx('padder', padderClassName)}>{children}</div>
        </HorizontalHiddenScroll>
    );
};

export default HorizontalScroller;
