import React from 'react';
import {useInView} from 'react-intersection-observer';

import {IWithClassName} from 'types/withClassName';
import {IHotelsCrossSale} from 'server/api/HotelsCrossSaleApi/types/IHotelsCrossSale';
import IGetCrossSaleHotelsResponseV1 from 'server/api/HotelsCrossSaleApi/types/IGetCrossSaleHotelsResponseV1';

import {loadable} from 'utilities/componentLoadable';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as aviaCrossSaleKeyset from 'i18nNew/avia-crossSale';

import Link from 'components/Link/Link';
import ButtonLink from 'components/ButtonLink/ButtonLink';

import cx from './HotelsCrossSaleMap.scss';

export interface ICrossSaleMapProps extends IWithClassName, IWithQaAttributes {
    mapClassName?: string;
    crossSaleData: IGetCrossSaleHotelsResponseV1 | IHotelsCrossSale;
    linkUrl: string;
    showMapOnIntersection?: boolean;

    handleLinkClick?(): void;
}

const HotelsSearchStaticMap = loadable(
    () => import('components/HotelsSearchStaticMap/HotelsSearchStaticMap'),
);

const HotelsCrossSaleMap: React.FC<ICrossSaleMapProps> = props => {
    const {
        className,
        mapClassName,
        linkUrl,
        showMapOnIntersection,
        crossSaleData: {bboxAsStruct, hotels},
        handleLinkClick,
    } = props;

    const {isMobile} = useDeviceType();
    const [ref, wasInView] = useInView({
        triggerOnce: true,
        skip: !showMapOnIntersection,
    });

    return (
        <div
            className={cx('root', className)}
            ref={ref}
            {...prepareQaAttributes(props)}
        >
            <div className={cx('map')}>
                {(!showMapOnIntersection || wasInView) && (
                    <HotelsSearchStaticMap
                        className={mapClassName}
                        hotels={hotels}
                        bboxAsStruct={bboxAsStruct}
                        activeHotelPermalink={
                            isMobile ? undefined : hotels[0]?.hotel.permalink
                        }
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'map',
                        })}
                    />
                )}
                <Link
                    className={cx('link')}
                    to={linkUrl}
                    target="_blank"
                    rel="noopener noreferrer"
                    onClick={handleLinkClick}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'searchLink',
                    })}
                />
            </div>

            {isMobile && (
                <div className={cx('buttonWrapper')}>
                    <ButtonLink
                        to={linkUrl}
                        onClick={handleLinkClick}
                        target="_blank"
                        rel="noopener noreferrer"
                        theme="raised"
                        width="max"
                        size="l"
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'buttonSearchLink',
                        })}
                    >
                        {aviaCrossSaleKeyset.selectHotel()}
                    </ButtonLink>
                </div>
            )}
        </div>
    );
};

export default HotelsCrossSaleMap;
