import React, {ReactNode, PureComponent} from 'react';
import {renderToString} from 'react-dom/server';
import {IYMapsApi} from 'react-yandex-maps';

/* Types */
import {IWithClassName} from 'types/withClassName';
import {IHotelWithMinPrice} from 'types/hotels/hotel/IHotelWithMinPrice';
import {IHotelWithOffers} from 'types/hotels/hotel/IHotelWithOffers';

import {formatPrice} from 'utilities/currency';
import IPrice from 'utilities/currency/PriceInterface';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import {YandexMapPlacemark} from 'components/YandexMaps/YandexMaps';

import cx from './HotelsMapStaticMarker.scss';

interface IHotelsMapStaticMarkerProps
    extends IWithClassName,
        IWithQaAttributes {
    hotel: IHotelWithOffers | IHotelWithMinPrice;
    isActiveMarker?: boolean;
    yandexMapAPIInstance: IYMapsApi | null;
    onSetMarkerPosition?: (position: {
        top: number;
        left: number;
        bottom: number;
    }) => void;
}

const HOVER_Z_INDEX = 10;
const HAS_PRICE_Z_INDEX = 9;
const WITHOUT_PRICE_Z_INDEX = 1;

class HotelsMapStaticMarker extends PureComponent<IHotelsMapStaticMarkerProps> {
    private getPrice = (): IPrice | undefined => {
        const {hotel} = this.props;

        return 'minPrice' in hotel ? hotel.minPrice : hotel.offers?.[0]?.price;
    };

    private getMarkerZIndex = (): number => {
        const {isActiveMarker} = this.props;

        if (isActiveMarker) {
            return HOVER_Z_INDEX;
        }

        if (this.getPrice()) {
            return HAS_PRICE_Z_INDEX;
        }

        return WITHOUT_PRICE_Z_INDEX;
    };

    /* Render */

    private renderHotelMarker = (): React.ReactElement => {
        const {isActiveMarker} = this.props;

        const price = this.getPrice();

        if (!price) {
            return (
                <div
                    className={cx('markerWithoutPrice', {
                        ['markerWithPrice_active']: isActiveMarker,
                    })}
                    {...prepareQaAttributes(this.props)}
                />
            );
        }

        return (
            <div
                className={cx('markerWithPrice', {
                    ['markerWithPrice_active']: isActiveMarker,
                })}
                {...prepareQaAttributes(this.props)}
            >
                {formatPrice({
                    ...price,
                    isCurrencyShown: true,
                })}
            </div>
        );
    };

    render(): ReactNode {
        const {
            hotel: {
                hotel: {permalink, coordinates},
            },
            isActiveMarker,
            yandexMapAPIInstance,
            onSetMarkerPosition,
        } = this.props;
        const hotelMarkerNode = this.renderHotelMarker();

        if (!yandexMapAPIInstance || !hotelMarkerNode) return null;

        const markerContentLayout =
            yandexMapAPIInstance.templateLayoutFactory.createClass(
                renderToString(hotelMarkerNode),
                {
                    build: function () {
                        this.constructor.superclass.build.call(this);
                        this._markerNode = this.getElement().firstChild;
                        this._parentElementNode = this.getParentElement();
                    },
                    getShape: function () {
                        const {
                            offsetWidth: markerWidth,
                            offsetHeight: markerHeight,
                        } = this._markerNode;

                        // Добавил проверку на markerHeight чтобы избежать лишних апдейтов
                        // когда убираем маркер с карточки
                        if (
                            isActiveMarker &&
                            onSetMarkerPosition &&
                            markerHeight
                        ) {
                            const {offsetTop, offsetLeft} =
                                this._parentElementNode;

                            onSetMarkerPosition({
                                top: offsetTop - markerHeight,
                                left: offsetLeft,
                                bottom: offsetTop,
                            });
                        }

                        return new yandexMapAPIInstance.shape.Rectangle(
                            new yandexMapAPIInstance.geometry.pixel.Rectangle([
                                [markerWidth / 2, -markerHeight],
                                [-markerWidth / 2, 0],
                            ]),
                        );
                    },
                },
            );

        return (
            <YandexMapPlacemark
                key={permalink}
                geometry={[coordinates.lat, coordinates.lon]}
                options={{
                    iconLayout: markerContentLayout,
                    zIndex: this.getMarkerZIndex(),
                }}
            />
        );
    }
}

export default HotelsMapStaticMarker;
