import React from 'react';
import times from 'lodash/times';

import {IWithClassName} from 'types/withClassName';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import {CardWithDeviceLayout} from 'components/CardWithDeviceLayout/CardWithDeviceLayout';
import Heading from 'components/Heading/Heading';
import Intersperse from 'components/Intersperse/Intersperse';
import GgreenArrowS from 'icons/illustrations/GgreenArrow/GgreenArrowS';
import GreenArrowDownS from 'icons/illustrations/GreenArrowDown/GreenArrowDownS';

import cx from './HowToBuy.scss';

export type THowToBuyItem = {
    title: string;
    text: string;
};

interface IHowToBuyProps extends IWithClassName, IWithQaAttributes {
    title: string;
    items: THowToBuyItem[];
}

const HowToBuy: React.FC<IHowToBuyProps> = props => {
    const {className, title, items} = props;

    const deviceType = useDeviceType();

    const stepQa = prepareQaAttributes({
        parent: props,
        current: 'step',
    });

    return (
        <CardWithDeviceLayout
            className={cx('root', deviceMods('root', deviceType), className)}
        >
            <Heading
                className={cx('title')}
                level="2"
                {...prepareQaAttributes({
                    parent: props,
                    current: 'title',
                })}
            >
                {title}
            </Heading>

            <div className={cx('steps')}>
                <Intersperse
                    separator={
                        deviceType.isMobile ? (
                            <GreenArrowDownS
                                className={cx('arrow')}
                                width={24}
                                height={28}
                            />
                        ) : (
                            <GgreenArrowS
                                className={cx('arrow')}
                                width={36}
                                height={24}
                            />
                        )
                    }
                >
                    {times(4, index => (
                        <div
                            key={index}
                            className={cx('item')}
                            {...prepareQaAttributes({
                                parent: stepQa,
                                key: index,
                            })}
                        >
                            <Heading
                                className={cx('itemTitle')}
                                level="3"
                                {...prepareQaAttributes({
                                    parent: stepQa,
                                    current: 'title',
                                    key: index,
                                })}
                            >
                                {items[index].title}
                            </Heading>

                            <div
                                className={cx('itemText')}
                                {...prepareQaAttributes({
                                    parent: stepQa,
                                    current: 'description',
                                    key: index,
                                })}
                            >
                                {items[index].text}
                            </div>
                        </div>
                    ))}
                </Intersperse>
            </div>
        </CardWithDeviceLayout>
    );
};

export default React.memo(HowToBuy);
