import {ReactNode} from 'react';
import {Meta} from '@storybook/react/types-6-0';

import {TStorybookFunctionComponent} from 'storybook/types';

import Box from 'components/Box/Box';
import Heading from 'components/Heading/Heading';

import DeviceTypeContext from 'contexts/DeviceTypeContext';

import ImagesCarousel from './ImagesCarousel';

import cx from './__docs__/styles.scss';

export default {
    title: 'Example/ImagesCarousel',
    component: ImagesCarousel,
} as Meta;

const IMAGES = [
    {
        src: 'https://avatars.mds.yandex.net/get-altay/492546/2a0000015fdc1509311f2ce61d539f0187e4/orig',
    },
    {
        src: 'https://i.travelapi.com/hotels/1000000/70000/64100/64055/7f313de5_z.jpg',
    },
    {
        src: 'https://avatars.mds.yandex.net/get-avia/244263/2a0000015ca818f0a0bbc5d0c6affef72a6a/offer-desktop-wide',
    },
    {
        src: 'https://i.travelapi.com/hotels/1000000/70000/64100/64055/8f79e181_z.jpg',
    },
    {
        src: 'https://i.travelapi.com/hotels/1000000/70000/64100/64055/4b7144d1_z.jpg',
    },
    {
        src: 'https://i.travelapi.com/hotels/1000000/70000/64100/64055/bf761c89_z.jpg',
    },
];

const SIZED_IMAGES = [
    {
        src: 'https://avatars.mds.yandex.net/get-altay/492546/2a0000015fdc1509311f2ce61d539f0187e4/orig',
        size: {width: 800, height: 511},
    },
    {
        src: 'https://i.travelapi.com/hotels/1000000/70000/64100/64055/7f313de5_z.jpg',
        size: {width: 1000, height: 667},
    },
    {
        src: 'https://avatars.mds.yandex.net/get-avia/244263/2a0000015ca818f0a0bbc5d0c6affef72a6a/offer-desktop-wide',
        size: {width: 1200, height: 350},
    },
    {
        src: 'https://avatars.mds.yandex.net/get-avia/233213/2a0000015a804658335a9ecdc825ab9bb534/offer-desktop-wide',
        size: {width: 800, height: 511},
    },
    {
        src: 'https://i.travelapi.com/hotels/1000000/70000/64100/64055/7f313de5_z.jpg',
        size: {width: 1000, height: 667},
    },
    {
        src: 'https://avatars.mds.yandex.net/get-avia/244263/2a0000015ca818f0a0bbc5d0c6affef72a6a/offer-desktop-wide',
        size: {width: 1200, height: 350},
    },
    {
        src: 'https://avatars.mds.yandex.net/get-altay/492546/2a0000015fdc1509311f2ce61d539f0187e4/orig',
        size: {width: 800, height: 511},
    },
    {
        src: 'https://i.travelapi.com/hotels/1000000/70000/64100/64055/7f313de5_z.jpg',
        size: {width: 1000, height: 667},
    },
    {
        src: 'https://avatars.mds.yandex.net/get-avia/244263/2a0000015ca818f0a0bbc5d0c6affef72a6a/offer-desktop-wide',
        size: {width: 1200, height: 350},
    },
];

export const Simple: TStorybookFunctionComponent = () => (
    <DeviceTypeContext.Consumer>
        {(deviceType): ReactNode => (
            <ImagesCarousel
                images={IMAGES}
                isDesktop={deviceType.isDesktop}
                height={200}
            />
        )}
    </DeviceTypeContext.Consumer>
);
Simple.parameters = {
    info: {
        comment: 'Вид скрола зависит от типа устроства (десктоп/тач)',
    },
};

export const Mini: TStorybookFunctionComponent = () => (
    <div style={{width: 180}}>
        <DeviceTypeContext.Consumer>
            {(deviceType): ReactNode => (
                <ImagesCarousel
                    images={IMAGES}
                    type="mini"
                    arrowsType="dark"
                    isDesktop={deviceType.isDesktop}
                    height={180}
                />
            )}
        </DeviceTypeContext.Consumer>
    </div>
);

export const WithClickHandler: TStorybookFunctionComponent = () => (
    <DeviceTypeContext.Consumer>
        {(deviceType): ReactNode => (
            <ImagesCarousel
                images={IMAGES}
                onImageClick={(src): void => alert(`Картинка с url: ${src}`)}
                isDesktop={deviceType.isDesktop}
                height={180}
            />
        )}
    </DeviceTypeContext.Consumer>
);

export const LazyLoad: TStorybookFunctionComponent = () => (
    <Box between={4}>
        <div>
            <Heading level={3}>Type: `full`</Heading>
            <ImagesCarousel
                images={SIZED_IMAGES}
                isDesktop
                height={200}
                rightOffset={20}
                lazyLoad
            />
        </div>
        <div>
            <div style={{width: 300}}>
                <Heading level={3}>Type: `mini`</Heading>
                <ImagesCarousel
                    images={SIZED_IMAGES}
                    isDesktop
                    height={300}
                    lazyLoad
                    type="mini"
                    arrowsType="dark"
                />
            </div>
        </div>
        <div>
            <Heading level={3}>Native scroll</Heading>
            <ImagesCarousel
                images={SIZED_IMAGES}
                isDesktop={false}
                height={300}
                rightOffset={20}
                lazyLoad
            />
        </div>
    </Box>
);
LazyLoad.parameters = {
    info: {
        comment:
            'Для lazyLoad необходима информация о размерах изображений, чтобы понимать сколько изображений нужно загружать сначала и подгружать при скролле',
    },
};

export const Spacing: TStorybookFunctionComponent = () => (
    <div style={{border: '1px solid'}}>
        <ImagesCarousel
            images={IMAGES}
            isDesktop
            imageClassName={cx('withBorderRadius')}
            height={300}
            spaceBetween={1}
            leftOffset={32}
            rightOffset={32}
            spaceBefore={8}
            spaceAfter={8}
        />
    </div>
);
Spacing.parameters = {
    info: {
        comment: 'Различные отступы задаются аналогично базовой карусели',
    },
};

export const Gradient: TStorybookFunctionComponent = () => (
    <div style={{height: 300}}>
        <ImagesCarousel
            images={SIZED_IMAGES}
            spaceBetween={1}
            gradient
            isDesktop
        />
    </div>
);

export const EqualHeightContainer: TStorybookFunctionComponent = () => (
    <Box between={4}>
        <div>
            <Heading level={3}>Type: `mini`.</Heading>
            <div className={cx('card')}>
                <div style={{width: 200}}>
                    <ImagesCarousel
                        images={IMAGES}
                        type="mini"
                        arrowsType="dark"
                        spaceBetween={0}
                        isDesktop
                    />
                </div>

                <div style={{height: 200}}></div>
            </div>
        </div>

        <div>
            <Heading level={3}>
                Type: `mini`. If image cropped vertically, add prop `autoHeight:
                true`
            </Heading>
            <div style={{display: 'flex'}}>
                <div style={{width: 'calc(100% - 300px)'}}>
                    <ImagesCarousel
                        images={IMAGES}
                        type="mini"
                        autoHeight
                        isDesktop
                    />
                </div>

                <div style={{width: 300, paddingLeft: 20}}>
                    <p>
                        Lorem ipsum dolor sit amet, consectetur adipiscing elit.
                        In vitae elementum arcu. Phasellus dictum molestie sem,
                        eu volutpat est luctus sed. Integer tempor convallis
                        velit ac iaculis. Nam lobortis, augue at rhoncus rutrum,
                        sem ligula tincidunt augue, eget fermentum nisi magna ut
                        purus. Nullam accumsan diam dolor, at viverra nisl
                        tempus eu. Etiam a aliquet felis, ac pellentesque ante.
                        Aliquam augue augue, egestas eget felis eu, tincidunt
                        egestas dolor. Nullam maximus consequat pellentesque.
                    </p>
                </div>
            </div>
        </div>

        <div>
            <Heading level={3}>Type: `full`. Need autoHeight prop</Heading>
            <div style={{display: 'flex'}}>
                <div style={{width: '80%'}}>
                    <ImagesCarousel images={IMAGES} autoHeight isDesktop />
                </div>

                <div style={{width: '20%', paddingLeft: 20}}>
                    <p>
                        Lorem ipsum dolor sit amet, consectetur adipiscing elit.
                        In vitae elementum arcu. Phasellus dictum molestie sem,
                        eu volutpat est luctus sed. Integer tempor convallis
                        velit ac iaculis. Nam lobortis, augue at rhoncus rutrum,
                        sem ligula tincidunt augue, eget fermentum nisi magna ut
                        purus. Nullam accumsan diam dolor, at viverra nisl
                        tempus eu. Etiam a aliquet felis, ac pellentesque ante.
                    </p>
                </div>
            </div>
        </div>
    </Box>
);

export const WideContainer: TStorybookFunctionComponent = () => (
    <DeviceTypeContext.Consumer>
        {(deviceType): ReactNode => (
            <ImagesCarousel images={IMAGES} isDesktop={deviceType.isDesktop} />
        )}
    </DeviceTypeContext.Consumer>
);
