import React, {useMemo, useState, useCallback} from 'react';

import {TImagesSelectorImage} from '../../types/IImagesSelectorImage';
import {IWithClassName} from 'types/withClassName';
import EAsyncStatus from 'types/common/EAsyncStatus';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import Button from 'components/Button/Button';
import CloseIcon from 'icons/12/Close';
import Flex from 'components/Flex/Flex';
import Spinner from 'components/Spinner/Spinner';

import cx from './SelectedImage.scss';

interface ISelectedImageProps extends IWithClassName {
    image: TImagesSelectorImage;
    onDeleteClick: (image: TImagesSelectorImage) => void;
}

const SelectedImage: React.FC<ISelectedImageProps> = ({
    className,
    image,
    onDeleteClick,
}) => {
    const deviceType = useDeviceType();
    const [dataUrl, setDataUrl] = useState<string>();
    const imageSrc = useMemo(() => {
        if ('src' in image) {
            return image.src;
        }

        if (image.file) {
            if (dataUrl) {
                return dataUrl;
            }

            const reader = new FileReader();

            reader.readAsDataURL(image.file);

            reader.onload = (e: ProgressEvent<FileReader>): void => {
                if (e.target) {
                    setDataUrl(e.target.result as string);
                }
            };
        }
    }, [image, dataUrl]);
    const handleDeleteClick = useCallback(
        () => onDeleteClick(image),
        [onDeleteClick, image],
    );

    return (
        <div
            className={cx('root', deviceMods('root', deviceType), className)}
            style={{backgroundImage: `url(${imageSrc})`}}
        >
            {'status' in image && image.status === EAsyncStatus.LOADING && (
                <>
                    <div className={cx('fade')} />
                    <Flex
                        className={cx('spinnerWrap')}
                        alignItems="center"
                        justifyContent="center"
                    >
                        <Spinner className={cx('spinner')} />
                    </Flex>
                </>
            )}
            <Flex // для увеличения области клика
                onClick={handleDeleteClick}
                className={cx('buttonWrap')}
                alignItems="center"
                justifyContent="center"
            >
                <Button
                    className={cx('button')}
                    size="s"
                    theme="plain"
                    icon={<CloseIcon />}
                />
            </Flex>
        </div>
    );
};

export default SelectedImage;
