import React, {ReactNode, useCallback} from 'react';

import {IWithClassName} from 'types/withClassName';
import {TResponsiveImage} from 'types/common/TImage';

import Swipeable, {
    ISwipeableRenderItemParams,
} from 'components/Swipeable/Swipeable';
import Navigation from './components/Navigation/Navigation';
import Image from 'components/TravelImage/TravelImage';

import cx from './ImagesSwipeCarousel.scss';

interface IImagesSnapCarouselProps extends IWithClassName {
    images: TResponsiveImage[];

    maxItemsCount?: number;
    borderRadius?: 'xs' | 's' | 'm' | 'l';
    borderWidth?: 's';

    onScrollContent?: (index?: number) => void;
    onImageClick?: () => void;
    swipeableStyle?: React.CSSProperties;
    swipeableClassName?: string;
}

const ImagesSwipeCarousel: React.FC<IImagesSnapCarouselProps> = ({
    images,
    borderRadius,
    maxItemsCount = 5,
    borderWidth,
    swipeableStyle,
    className,
    swipeableClassName,
    onImageClick,
    onScrollContent,
}) => {
    const renderItem = useCallback(
        (image: ISwipeableRenderItemParams<TResponsiveImage>): ReactNode => (
            <Image
                className={cx('imageContainer')}
                imageClassName={cx('image')}
                src={image.data.src}
                isWide
                responsiveSet={image.data.additionalSizes}
                hasFitCover
                onClick={onImageClick}
            />
        ),
        [onImageClick],
    );

    const renderNav = useCallback(
        (items, currentIndex) => (
            <Navigation
                itemsCount={items.length}
                maxItemsCount={maxItemsCount}
                activeItem={currentIndex}
            />
        ),
        [maxItemsCount],
    );

    return (
        <div className={cx('root', className)}>
            <Swipeable
                className={cx(
                    'swipeable',
                    {
                        [`swipeable_borderRadius_${borderRadius}`]:
                            borderRadius,
                        [`swipeable_border_${borderWidth}`]: borderWidth,
                    },
                    swipeableClassName,
                )}
                style={swipeableStyle}
                itemsData={images}
                sideThreshold={0.1}
                showNav
                onSideSwipe={onScrollContent}
                spacing="xs"
                renderItem={renderItem}
                renderNav={renderNav}
                itemVerticalAlignment="stretch"
                disableSwipe={images.length === 1}
            />
        </div>
    );
};

export default ImagesSwipeCarousel;
