import {Meta} from '@storybook/react/types-6-0';
import {noop} from 'lodash';

import {TStorybookFunctionComponent} from 'storybook/types';

import Box from 'components/Box/Box';
import Heading from 'components/Heading/Heading';
import StyleguidistSizeTable from 'components/StyleguidistSizeTable/StyleguidistSizeTable';
import StyleguidistPropTable from 'components/StyleguidistPropTable/StyleguidistPropTable';
import ArrowLeftIcon from 'icons/24/ArrowLeft';
import KebabVerticalIcon from 'icons/16/KebabVertical';

import Intent, {EIntentTheme} from './Intent';

export default {
    title: 'Example/Intent',
    component: Intent,
} as Meta;

const INTENT_THEMES: EIntentTheme[] = [
    EIntentTheme.PRIMARY,
    EIntentTheme.ALERT,
];

export const Simple: TStorybookFunctionComponent = () => (
    <Box between="2">
        <Heading level="3">Однострочный</Heading>
        {INTENT_THEMES.map(theme => (
            <Box key={theme} between="5" inline>
                <Intent onClick={noop} text="Интентик" theme={theme} />
                <Intent onClick={noop} text="Интентик" theme={theme} />
            </Box>
        ))}
        <Heading level="3">С комментарием</Heading>
        {INTENT_THEMES.map(theme => (
            <Box key={theme} between="5" inline>
                <Intent
                    onClick={noop}
                    size="m-inset"
                    text="Интентик"
                    comment="Комментарий"
                    theme={theme}
                />
                <Intent
                    onClick={noop}
                    size="m-inset"
                    text="Интентик"
                    comment="Комментарий"
                    theme={theme}
                />
            </Box>
        ))}
    </Box>
);

export const State: TStorybookFunctionComponent = () => (
    <StyleguidistPropTable
        props={[
            {
                name: 'checked',
                props: {isChecked: true},
                example: <Intent onClick={noop} isChecked text="Интентик" />,
            },
            {
                name: 'disabled',
                props: {isDisabled: true},
                example: <Intent onClick={noop} isDisabled text="Интентик" />,
            },
        ]}
    />
);

export const Size: TStorybookFunctionComponent = () => (
    <StyleguidistSizeTable
        sizes={[
            {
                name: 's',
                size: '28px',
                example: <Intent size="s" text="Размер s" onCancel={noop} />,
            },
            {
                name: 'm',
                size: '36px',
                comment: 'По умолчанию',
                example: <Intent size="m" text="Размер m" onCancel={noop} />,
            },
            {
                name: 'm-inset',
                size: '36px',
                comment: 'Может иметь комментарий',
                example: (
                    <>
                        <Intent
                            size="m-inset"
                            text="Размер m-inset"
                            onCancel={noop}
                        />
                        <Intent
                            size="m-inset"
                            text="Размер m-inset"
                            comment="Комментарий"
                            onCancel={noop}
                        />
                    </>
                ),
            },
            {
                name: 'l',
                size: '44px/52px',
                comment: 'Может иметь комментарий',
                example: (
                    <>
                        <Intent size="l" text="Размер l" onCancel={noop} />
                        <Intent
                            size="l"
                            text="Размер l"
                            comment="Комментарий"
                            onCancel={noop}
                        />
                    </>
                ),
            },
            {
                name: 'xl',
                size: '52px',
                example: <Intent size="xl" text="Размер xl" onCancel={noop} />,
            },
        ]}
    />
);

export const Width: TStorybookFunctionComponent = () => (
    <Box between="2">
        <Box between="5" inline>
            <Intent size="m-inset" text="width = auto" onClick={noop} />
            <Intent
                size="m-inset"
                text="width = auto"
                comment="С комментарием"
                onClick={noop}
            />
        </Box>
        <Box between="1">
            <Intent
                width="max"
                size="m-inset"
                text="width = max"
                onClick={noop}
            />
            <Intent
                width="max"
                size="m-inset"
                text="width = auto"
                comment="С комментарием"
                onClick={noop}
            />
        </Box>
        <Box style={{width: '200px'}} between="1">
            <Intent
                width="max"
                size="m-inset"
                text="width = max width = max width = max"
                onClick={noop}
            />
            <Intent
                width="max"
                size="m-inset"
                text="width = auto width = max width = max"
                comment="С комментарием С комментарпием С комментарием"
                onClick={noop}
            />
        </Box>
    </Box>
);
Width.parameters = {
    info: {
        comment: (
            <>
                Можно задать ширину интента width = max | auto <br />
                По умолчанию auto (по содержимому) <br />В случае ширины max
                (или передачи в стилях для className максимальной ширины) текст
                может скрываться ...
            </>
        ),
    },
};

export const Handlers: TStorybookFunctionComponent = () => (
    <Box between="2">
        <Heading level="3">Однострочный</Heading>
        {INTENT_THEMES.map(theme => (
            <Box key={theme} between="5" inline>
                <Intent onClick={noop} text="Интентик" theme={theme} />
                <Intent onCancel={noop} text="Интентик" theme={theme} />
            </Box>
        ))}
        <Heading level="3">С комментарием</Heading>
        {INTENT_THEMES.map(theme => (
            <Box key={theme} between="5" inline>
                <Intent
                    onClick={noop}
                    size="m-inset"
                    text="Интентик"
                    comment="Комментарий"
                    theme={theme}
                />
                <Intent
                    onCancel={noop}
                    size="m-inset"
                    text="Интентик"
                    comment="Комментарий"
                    theme={theme}
                />
                <Intent
                    onCancel={noop}
                    onClick={noop}
                    size="m-inset"
                    text="Интентик"
                    comment="Комментарий"
                    theme={theme}
                />
            </Box>
        ))}
    </Box>
);
Handlers.parameters = {
    info: {
        comment: (
            <>
                Можно передать в интент либо функцию onClick <br />И в таком
                случае клик по интенту будет вызывать это действие <br />
                Либо передать onCancel и в таком случае можно будет только
                нажать на крестик (иконку) <br />
                Либо передать оба параметра вместе и в таком случае весь интент
                будет кликабельным Реализацию с отменой нужна для выбора тегов в
                поиске (планируется в поиске отелей)
            </>
        ),
    },
};

export const WithIcon: TStorybookFunctionComponent = () => (
    <Box between="2">
        <Heading level="3">С иконкой размера m</Heading>
        <Box>
            {INTENT_THEMES.map(theme => (
                <Intent
                    key={theme}
                    onClick={noop}
                    icon={<KebabVerticalIcon />}
                    theme={theme}
                />
            ))}
        </Box>
        <Heading level="3">С иконкой размера l</Heading>
        <Box>
            {INTENT_THEMES.map(theme => (
                <Intent
                    key={theme}
                    size="l"
                    onClick={noop}
                    icon={<ArrowLeftIcon />}
                    theme={theme}
                />
            ))}
        </Box>
    </Box>
);
WithIcon.parameters = {
    info: {
        comment: (
            <>
                В данные момент реализовано так, что можно передать либо иконку,
                либо текст с комменатрием <br />
                Иконка должна быть в размерах 16х16 (чтобы кнопка стала круглой)
            </>
        ),
    },
};
