import React, {useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import Box, {TBoxSizes} from 'components/Box/Box';
import {Intent} from 'components/Intent/Intent';
import HorizontalScroller from 'components/HorizontalScroller/HorizontalScroller';
import KebabVerticalIcon from 'icons/16/KebabVertical';

import cx from './IntentsList.scss';

export interface IBaseIntentItem {
    text: string;
    comment?: string;
}

export interface IIntentsListProps<IIntentItem extends IBaseIntentItem>
    extends IWithClassName {
    itemClassName?: string;
    items: IIntentItem[];
    onIntentClick: (item: IIntentItem) => void;

    isCollapsed?: boolean;
    collapsedCount?: number;
    onExpandButtonClick?: () => void;

    nowrap?: boolean;
    horizontalOffset?: TBoxSizes;
}

function IntentsList<IIntentItem extends IBaseIntentItem>({
    className,
    itemClassName,
    items,
    onIntentClick,
    isCollapsed,
    collapsedCount,
    onExpandButtonClick,
    nowrap,
    horizontalOffset,
    ...rest
}: IIntentsListProps<IIntentItem>): React.ReactElement {
    const Container = nowrap ? HorizontalScroller : 'div';
    const intentItems = useMemo(
        () =>
            isCollapsed && collapsedCount
                ? items.slice(0, collapsedCount)
                : items,
        [items, isCollapsed, collapsedCount],
    );

    return (
        <Container
            className={cx('root', {root_nowrap: nowrap}, className)}
            offset={horizontalOffset || 0}
        >
            <Box between="1" nowrap={nowrap} inline>
                {intentItems.map((item, index) => {
                    const intentTypeProps = item.comment
                        ? {
                              comment: item.comment,
                              size: 'm-inset' as 'm-inset',
                          }
                        : {
                              size: 'l' as 'l',
                          };

                    return (
                        <Intent
                            className={cx('item', itemClassName)}
                            key={index}
                            text={item.text}
                            width="auto"
                            onClick={() => onIntentClick(item)}
                            {...intentTypeProps}
                            {...prepareQaAttributes({
                                parent: rest,
                                current: 'intent',
                                key: index,
                            })}
                        />
                    );
                })}
                {intentItems.length < items.length &&
                    isCollapsed &&
                    onExpandButtonClick && (
                        <Intent
                            className={cx('item', itemClassName)}
                            icon={<KebabVerticalIcon />}
                            size="l"
                            width="auto"
                            onClick={onExpandButtonClick}
                        />
                    )}
            </Box>
        </Container>
    );
}

export default IntentsList;
