import React, {FunctionComponent, ReactElement, ReactNode} from 'react';

import cx from './Intersperse.scss';

function isReactElement(elem: any): elem is ReactElement {
    return Boolean(elem?.type && elem?.props);
}

const intersperseElementsArray = (
    arr: ReactNode[],
    separatorFn: (idx: number) => ReactNode,
): ReactNode[] => {
    const result: ReactNode[] = [];

    arr.forEach((elem, idx) => {
        result.push(elem);

        const isLast = idx === arr.length - 1;

        if (!isLast) {
            result.push(separatorFn(idx));
        }
    });

    return result;
};

type TIntersperseSeparator = ((idx: number) => ReactNode) | ReactNode;

interface IIntersperseProps {
    separator: TIntersperseSeparator;
    children?: ReactNode;
}

function getSeparatorFn(separator: TIntersperseSeparator) {
    if (typeof separator === 'function') {
        return separator as (idx: number) => ReactNode;
    }

    if (isReactElement(separator)) {
        return (idx: number) =>
            React.cloneElement(separator, {
                className: cx('separator', separator.props.className),
                idx,
            });
    }

    return (): TIntersperseSeparator => separator;
}

const Intersperse: FunctionComponent<IIntersperseProps> = props => {
    const childrenArr = React.Children.toArray(props.children).map(child => {
        /*
         * Оборачиваем в span, чтобы для разделителей работали селекторы :first-child, :last-child
         */
        if (typeof child === 'string' || typeof child === 'number') {
            return <span>{child}</span>;
        }

        return child;
    });

    const separatorFn = getSeparatorFn(props.separator);

    const interspersed = intersperseElementsArray(childrenArr, separatorFn);

    return React.createElement(React.Fragment, {}, ...interspersed);
};

export default Intersperse;
