import {ReactNode, ComponentType, forwardRef} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IIconProps} from 'icons/types/icon';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import TextWithIcon, {TSize} from 'components/TextWithIcon/TextWithIcon';

import cx from './Label.scss';

const TEXT_WITH_ICON_SIZE_MAP: Record<TLabelSizeType, TSize> = {
    s: 'xs',
    's-inset': 'xs',
    m: 's',
    'm-inset': 's',
};

export enum ELabelThemeType {
    PRIMARY = 'primary',
    NORMAL = 'normal',
    ATTENTION = 'attention',
    NEUTRAL = 'neutral',
    BRAND = 'brand',
    BLACK = 'black',
    /** TODO: Нужно добавить в гайд и дать более общее название */
    AEROFLOT = 'aeroflot',
}

export enum ECornersType {
    DEFAULT = 'default',
    BOTTOM = 'bottom',
    BOTTOM_RIGHT = 'bottomRight',
    ROUNDED = 'rounded',
}

export type TLabelSizeType = 's' | 's-inset' | 'm' | 'm-inset';

export interface ILabelProps extends IWithClassName, IWithQaAttributes {
    size?: TLabelSizeType;
    /**
     * Предопределенная тема, если null, то темы нет
     */
    theme?: ELabelThemeType;
    children: ReactNode;
    iconLeft?: ComponentType<IIconProps>;

    cornersType?: ECornersType;

    onClick?: () => void;
}

const Label = forwardRef<HTMLSpanElement, ILabelProps>(function LabelComponent(
    {
        className,
        children,
        size = 's',
        theme,
        cornersType = ECornersType.DEFAULT,
        onClick,
        iconLeft,
        ...rest
    },
    ref,
) {
    return (
        <span
            className={cx(
                'label',
                `label_size_${size}`,
                `label_corner_${cornersType}`,
                theme && `label_theme_${theme}`,
                className,
            )}
            ref={ref}
            onClick={onClick}
            {...prepareQaAttributes(rest)}
        >
            <TextWithIcon
                className={cx('textWithIcon')}
                text={children}
                size={TEXT_WITH_ICON_SIZE_MAP[size]}
                iconLeft={iconLeft}
                spaceBetween="1"
                iconSize={12}
            />
        </span>
    );
});

export default Label;
