import React, {useMemo} from 'react';

import {EProjectName} from 'constants/common';

import {EFooterProject} from 'components/Footer/types';
import {IWithClassName} from 'types/withClassName';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18nNew/happyPage';

import LayoutDefault from 'components/Layouts/LayoutDefault/LayoutDefault';
import Container from 'components/Layouts/Container/Container';
import Flex from 'components/Flex/Flex';
import SupportPhone from 'components/SupportPhone/SupportPhone';
import Box from 'components/Box/Box';
import BookLoader from 'components/BookLoader/BookLoader';
import FieldLabel from 'components/FieldLabel/FieldLabel';

import cx from './HappyPageLayout.scss';

interface IHappyPageLayoutProps extends IWithClassName {
    headerClassName?: string;
    project:
        | EProjectName.TRAINS
        | EProjectName.HOTELS
        | EProjectName.AVIA
        | EProjectName.BUSES;
    footerType?: EFooterProject;
    header?: React.ReactNode;
    banner?: React.ReactNode;
    orderHeaderInfo?: React.ReactNode;
    orderInfo?: React.ReactNode;
    supportTitle?: string;
    supportPhone?: React.ReactNode;
    orderActions?: React.ReactNode;
    crossSales?: React.ReactNode;
    crossSalesClassName?: string;
    isLoading?: boolean;
    error?: React.ReactNode;
}

const HappyPageLayout: React.FC<IHappyPageLayoutProps> = props => {
    const {
        className,
        headerClassName,
        project,
        header,
        banner,
        orderHeaderInfo,
        orderInfo,
        supportTitle = i18nBlock.supportTitle(),
        supportPhone = <SupportPhone />,
        orderActions,
        crossSales,
        crossSalesClassName,
        footerType,
        isLoading,
        error,
    } = props;

    const deviceType = useDeviceType();

    const content = useMemo(() => {
        if (isLoading) {
            return (
                <BookLoader
                    title={i18nBlock.loaderTitle()}
                    description={i18nBlock.loaderDescription()}
                    isLoading
                />
            );
        }

        return (
            <>
                <Container className={cx('container')}>
                    {header && (
                        <div className={cx('header', headerClassName)}>
                            {header}
                        </div>
                    )}

                    {banner && <div>{banner}</div>}

                    {orderHeaderInfo && (
                        <Box
                            below={deviceType.isMobile ? 2 : 4}
                            above={deviceType.isMobile ? 2 : 4}
                        >
                            {orderHeaderInfo}
                        </Box>
                    )}

                    {orderInfo && (
                        <div className={cx('orderInfo')}>{orderInfo}</div>
                    )}

                    {orderInfo && (
                        <Flex
                            flexDirection={
                                deviceType.isMobile ? 'column-reverse' : 'row'
                            }
                            justifyContent="space-between"
                            alignItems={
                                deviceType.isMobile ? 'initial' : 'center'
                            }
                            above={deviceType.isMobile ? 3 : 5}
                        >
                            <FieldLabel
                                label={supportTitle}
                                labelColor={
                                    deviceType.isMobile
                                        ? 'secondary'
                                        : 'primary'
                                }
                            >
                                <span className={cx('supportPhone')}>
                                    {supportPhone}
                                </span>
                            </FieldLabel>

                            <div className={cx('actions')}>{orderActions}</div>
                        </Flex>
                    )}

                    {error}
                </Container>

                {crossSales && (
                    <Container
                        className={cx('crossSales', crossSalesClassName)}
                    >
                        {crossSales}
                    </Container>
                )}
            </>
        );
    }, [
        banner,
        header,
        orderHeaderInfo,
        orderInfo,
        orderActions,
        crossSales,
        crossSalesClassName,
        isLoading,
        deviceType,
        headerClassName,
        error,
        supportPhone,
        supportTitle,
    ]);

    return (
        <LayoutDefault
            className={cx('root', deviceMods('root', deviceType), className)}
            contentClassName={cx('content', {content_loading: isLoading})}
            footerClassName={cx('footer')}
            project={project}
            footerType={footerType}
            isFixedNavigation
            showNavigation
            hasSideSheetNavigation
        >
            {content}
        </LayoutDefault>
    );
};

export default React.memo(HappyPageLayout);
