import React from 'react';

import {EProjectName} from 'constants/common';

import {IWithClassName} from 'types/withClassName';
import {EFooterProject} from 'components/Footer/types';
import EHeaderBorderBottomType from 'components/Header/types/EHeaderBorderBottomType';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Header from 'components/Header/Header';
import Footer from 'components/Footer/Footer';

import DeviceTypeContext from 'contexts/DeviceTypeContext';

import cx from './LayoutDefault.scss';

export enum ELayoutBgColor {
    BASIC = 'basic',
    LIGHT = 'light',
    DARK = 'dark',
}

export interface ILayoutDefaultProps extends IWithClassName, IWithQaAttributes {
    headerClassName?: string;
    headerWrapperClassName?: string;
    headerBorderBottomType?: EHeaderBorderBottomType;
    formWrapperClassName?: string;
    contentClassName?: string;
    footerClassName?: string;
    footerWrapperClassName?: string;
    bgColor?: ELayoutBgColor;
    searchFormCollapsedClassName?: string;
    project?: EProjectName;
    footerType?: EFooterProject;
    showHeader?: boolean;
    showFooter?: boolean;
    showSearchForm?: boolean;
    showNavigation?: boolean;
    isFixedNavigation?: boolean;
    hasSideSheetNavigation?: boolean;
    searchFormIsStatic?: boolean;
    initiallyCalendarIsOpen?: boolean;
    validateSearchFormOnMount?: boolean;
    searchFormInitialIsExpanded?: boolean;
    children: React.ReactNode;
    filters?: React.ReactNode;
    hideTopMobileBlock?: boolean;
    hideSearchInformation?: boolean;
    onSearchFormSubmit?(): void;
}

const LayoutDefault: React.FC<ILayoutDefaultProps> = props => {
    const {
        className,
        headerClassName,
        headerWrapperClassName,
        headerBorderBottomType,
        formWrapperClassName,
        contentClassName,
        footerClassName,
        footerWrapperClassName,
        bgColor = ELayoutBgColor.LIGHT,
        searchFormCollapsedClassName,
        project,
        footerType,
        showHeader = true,
        showFooter = true,
        showSearchForm,
        showNavigation,
        isFixedNavigation,
        hasSideSheetNavigation,
        searchFormIsStatic,
        initiallyCalendarIsOpen,
        validateSearchFormOnMount,
        searchFormInitialIsExpanded,
        children,
        filters,
        hideTopMobileBlock,
        hideSearchInformation,
        onSearchFormSubmit,
    } = props;

    return (
        <DeviceTypeContext.Consumer>
            {(deviceType): React.ReactNode => (
                <div
                    className={cx(
                        'root',
                        {
                            root_mobile: deviceType.isMobile,
                        },
                        `root_bgColor_${bgColor}`,
                        className,
                    )}
                    {...prepareQaAttributes(props)}
                >
                    {showHeader && (
                        <Header
                            className={headerClassName}
                            wrapperClassName={headerWrapperClassName}
                            formWrapperClassName={formWrapperClassName}
                            searchFormCollapsedClassName={
                                searchFormCollapsedClassName
                            }
                            isFixed={isFixedNavigation}
                            showSearchForm={showSearchForm}
                            showNavigation={showNavigation}
                            hasSideSheetNavigation={hasSideSheetNavigation}
                            searchFormIsStatic={searchFormIsStatic}
                            initiallyCalendarIsOpen={initiallyCalendarIsOpen}
                            validateSearchFormOnMount={
                                validateSearchFormOnMount
                            }
                            searchFormInitialIsExpanded={
                                searchFormInitialIsExpanded
                            }
                            onSubmitForm={onSearchFormSubmit}
                            project={project}
                            filters={filters}
                            hideTopMobileBlock={hideTopMobileBlock}
                            hideSearchInformation={Boolean(
                                hideSearchInformation,
                            )}
                            borderBottomType={headerBorderBottomType}
                        />
                    )}

                    <div className={cx('content', contentClassName)}>
                        {children}
                    </div>

                    {showFooter && (
                        <Footer
                            className={footerClassName}
                            wrapperClassName={footerWrapperClassName}
                            project={project}
                            footerType={footerType}
                        />
                    )}
                </div>
            )}
        </DeviceTypeContext.Consumer>
    );
};

export default LayoutDefault;
