import React, {ReactNode, ComponentType} from 'react';

import {EProjectName} from 'constants/common';

import {IWithClassName} from 'types/withClassName';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18nNew/components-CheckOrderAuthorization';

import Flex from 'components/Flex/Flex';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import PaymentAdvantage from 'components/Layouts/PaymentLayout/components/PaymentAdvantage/PaymentAdvantage';
import HotelIcon from 'icons/16/Hotel';
import TrainIcon from 'icons/16/Train';
import FlightIcon from 'icons/16/Flight';
import BusIcon from 'icons/16/Bus';

import cx from './PaymentOrderInfo.scss';

function getOrderIcon(type: EProjectName): ComponentType | undefined {
    switch (type) {
        case EProjectName.TRAINS:
            return TrainIcon;
        case EProjectName.HOTELS:
            return HotelIcon;
        case EProjectName.AVIA:
            return FlightIcon;
        case EProjectName.BUSES:
            return BusIcon;
        default:
            return undefined;
    }
}

interface IPaymentOrderInfoProps extends IWithClassName {
    description: {
        general: React.ReactNode;
        additional: React.ReactNode;
    };
    orderId?: string;
    orderType: EProjectName;
    advantages?: ReactNode[];
}

const PaymentOrderInfo: React.FC<IPaymentOrderInfoProps> = props => {
    const {className, description, orderId, orderType, advantages} = props;

    const deviceType = useDeviceType();
    const {isDesktop, isMobile} = deviceType;
    const OrderIcon = getOrderIcon(orderType);

    return (
        <div className={cx(className, 'root', deviceMods('root', deviceType))}>
            <div className={cx('details')}>
                {isDesktop ? (
                    <>
                        <div className={cx('detailItem')}>
                            {description.general}
                        </div>
                        <div className={cx('detailItem')}>
                            {description.additional}
                        </div>
                    </>
                ) : (
                    <>
                        {description.general} {description.additional}
                    </>
                )}
            </div>
            {isDesktop && advantages && advantages.length ? (
                <Flex flexDirection="column" between={2} above={2} below={7}>
                    {advantages.map((advantage, index) => (
                        <PaymentAdvantage
                            key={
                                React.isValidElement(advantage)
                                    ? advantage.key ?? index
                                    : index
                            }
                        >
                            {advantage}
                        </PaymentAdvantage>
                    ))}
                </Flex>
            ) : null}
            {orderId && (
                <TextWithIcon
                    iconLeft={OrderIcon}
                    text={i18nBlock.orderIdTitle({orderId})}
                    className={cx('orderId')}
                    size="s"
                />
            )}
            {isMobile && advantages && advantages.length ? (
                <Flex flexDirection="column" between={2} above={2}>
                    {advantages.map((advantage, index) => (
                        <PaymentAdvantage
                            key={
                                React.isValidElement(advantage)
                                    ? advantage.key ?? index
                                    : index
                            }
                        >
                            {advantage}
                        </PaymentAdvantage>
                    ))}
                </Flex>
            ) : null}
        </div>
    );
};

export default PaymentOrderInfo;
