import React from 'react';

import {IWithClassName} from 'types/withClassName';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Flex from 'components/Flex/Flex';
import PaymentTitle from './components/PaymentTitle/PaymentTitle';
import PaymentDetails from './components/PaymentDetails/PaymentDetails';
import PaymentAdvantage from '../../components/PaymentAdvantage/PaymentAdvantage';

import cx from './PaymentOrderInfo.scss';

interface IPaymentOrderInfoProps extends IWithClassName, IWithQaAttributes {
    title: React.ReactNode;
    details: React.ReactNode[][];
    advantages?: React.ReactNode[];
}

const PaymentOrderInfo: React.FC<IPaymentOrderInfoProps> = props => {
    const {className, title, details, advantages = []} = props;

    const deviceType = useDeviceType();

    return (
        <div className={cx(className, 'root', deviceMods('root', deviceType))}>
            <PaymentTitle className={cx('amount')}>{title}</PaymentTitle>

            <Flex className={cx('details')} flexDirection="column" between={2}>
                {details.map((serviceDetails, serviceIndex) => (
                    <PaymentDetails
                        key={serviceIndex}
                        className={cx('serviceDetails')}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'paymentDetails',
                            key: serviceIndex,
                        })}
                    >
                        {serviceDetails.map(
                            (detail, index) =>
                                detail && (
                                    <span
                                        key={
                                            React.isValidElement(detail)
                                                ? detail.key ?? index
                                                : index
                                        }
                                    >
                                        {detail}
                                    </span>
                                ),
                        )}
                    </PaymentDetails>
                ))}
            </Flex>

            <Flex flexDirection="column" between={2}>
                {advantages.map((advantage, index) => (
                    <PaymentAdvantage
                        key={
                            React.isValidElement(advantage)
                                ? advantage.key ?? index
                                : index
                        }
                    >
                        {advantage}
                    </PaymentAdvantage>
                ))}
            </Flex>
        </div>
    );
};

export default PaymentOrderInfo;
