import {FC, AnchorHTMLAttributes, InputHTMLAttributes} from 'react';
import {
    Link as RouterLink,
    LinkProps as RouterLinkProps,
} from 'react-router-dom';
import classNames from 'classnames';

import {IWithClassName} from 'types/withClassName';

import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceModDesktop} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import {ILinkProps as LinkProps} from '@yandex-lego/components/Link/desktop';
import {LegoLinkPresenter} from 'components/Link/controls/LegoLinkPresenter';

import cx from './Link.scss';

export type TLinkThemeType =
    | 'normal'
    | 'alert'
    | 'black'
    | 'ghost'
    | 'outer'
    | 'brand'
    | 'light'
    | 'none';

export interface ILegoLinkProps
    extends LinkProps,
        IWithClassName,
        IWithQaAttributes {
    url: string;
    cls?: never;
    theme?: TLinkThemeType;
    text?: string;
    download?: boolean;
    controlAttrs?: InputHTMLAttributes<HTMLElement>;
}

export interface IRouterLinkProps
    extends Omit<RouterLinkProps, 'to'>,
        IWithClassName,
        IWithQaAttributes,
        AnchorHTMLAttributes<HTMLAnchorElement> {
    to?: RouterLinkProps['to'];
    text?: string;
    theme?: TLinkThemeType;
    download?: never;
    disabled?: boolean;
}

export type TLinkProps = ILegoLinkProps | IRouterLinkProps;

const Link: FC<TLinkProps> = props => {
    const deviceType = useDeviceType();
    const rootClassName = cx('root', deviceModDesktop('root', deviceType));

    const {children, text, theme = 'normal'} = props;

    if ('url' in props) {
        const {url, controlAttrs, className, ...rest} = props;

        return (
            <LegoLinkPresenter
                href={url}
                theme={theme as any}
                view="default"
                className={classNames(
                    rootClassName,
                    'Link_lego',
                    `Link_theme_${theme}`,
                    className,
                )}
                {...controlAttrs}
                {...rest}
                {...prepareQaAttributes(props)}
            >
                {children || text}
            </LegoLinkPresenter>
        );
    }

    const {to = '', className, ...rest} = props;

    return (
        <RouterLink
            className={classNames(
                rootClassName,
                'Link',
                'Link_router',
                `Link_theme_${theme}`,
                className,
            )}
            to={to}
            {...rest}
            {...prepareQaAttributes(props)}
        >
            {children || text}
        </RouterLink>
    );
};

export default Link;
