import React from 'react';

const DEFAULT_MS = 500;

interface IMinLoaderTimeProps {
    isLoaderNeeded: boolean;
    loaderNode: React.ReactNode;
    componentNode: React.ReactNode;
    minLoaderTime?: number;
}

interface IMinLoaderTimeState {
    shouldRenderLoader: boolean;
}

/** Компонент для задания минимального времени отображения лоадера */
class MinLoaderTime extends React.Component<
    IMinLoaderTimeProps,
    IMinLoaderTimeState
> {
    private _animationTimeout: any = null;

    readonly state: IMinLoaderTimeState = {
        shouldRenderLoader: this.props.isLoaderNeeded,
    };

    componentDidMount(): void {
        const {isLoaderNeeded} = this.props;

        if (isLoaderNeeded) {
            this.startAnimationTimeout();
        }
    }

    componentDidUpdate(prevProps: Readonly<IMinLoaderTimeProps>): void {
        this.checkChangeIsLoaderNeeded(prevProps);
    }

    componentWillUnmount(): void {
        this.clearAnimationTimeout();
    }

    private startAnimationTimeout = (): void => {
        const {minLoaderTime = DEFAULT_MS} = this.props;

        this._animationTimeout = setTimeout(
            this.finishAnimationTimeout,
            minLoaderTime,
        );
    };

    private finishAnimationTimeout = (): void => {
        const {isLoaderNeeded} = this.props;

        if (!isLoaderNeeded) {
            this.hideLoaderComponent();
        }

        this.clearAnimationTimeout();
    };

    private hideLoaderComponent(): void {
        this.setState({shouldRenderLoader: false});
    }

    private showLoaderComponent(): void {
        this.setState({shouldRenderLoader: true});
    }

    private clearAnimationTimeout(): void {
        clearTimeout(this._animationTimeout);

        this._animationTimeout = null;
    }

    private checkChangeIsLoaderNeeded(
        prevProps: Readonly<IMinLoaderTimeProps>,
    ): void {
        const {isLoaderNeeded} = this.props;

        if (
            prevProps.isLoaderNeeded &&
            !isLoaderNeeded &&
            this._animationTimeout === null
        ) {
            this.hideLoaderComponent();
        }

        if (!prevProps.isLoaderNeeded && isLoaderNeeded) {
            this.startAnimationTimeout();
            this.showLoaderComponent();
        }
    }

    render(): React.ReactNode {
        const {shouldRenderLoader} = this.state;
        const {loaderNode, componentNode} = this.props;

        return <>{shouldRenderLoader ? loaderNode : componentNode}</>;
    }
}

export default MinLoaderTime;
