import {
    Component,
    ReactNode,
    ReactElement,
    ElementType,
    Ref,
    RefObject,
    SyntheticEvent,
    createRef,
    forwardRef,
} from 'react';
import ReactDOM from 'react-dom';
import noop from 'lodash/noop';
import FocusLock, {AutoFocusInside} from 'react-focus-lock';

import {IWithClassName} from 'types/withClassName';

import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import scrollTo from 'utilities/dom/scrollTo';

import Box, {IBoxProps} from 'components/Box/Box';
import Card, {ICardProps} from 'components/Card/Card';
import HideBodyVerticalScroll from 'components/HideBodyVerticalScroll/HideBodyVerticalScroll';
import CloseIcon from 'icons/16/Close';
import {ModalContext} from './components/ModalContext/ModalContext';

import LayersContext from 'contexts/LayersContext';
import ScopeContext from 'contexts/ScopeContext';

import cx from './Modal.scss';

export const MODAL_LAYER_MULTIPLIER = 1000;

const CardWithForwardRef = forwardRef(
    (props: ICardProps, ref: Ref<HTMLElement>) => {
        return <Card {...props} innerRef={ref} />;
    },
);

CardWithForwardRef.displayName = 'CardWithForwardRef';

export interface IModalProps extends IWithClassName, IWithQaAttributes {
    /**
     * Контент модалки. Для полной стилизации необходимо контент
     * обернуть в `<Modal.Content>{children}</Modal.Content>`
     */
    children?: ReactNode;

    /**
     * Компонент, который будет использоваться в качестве обертки для FocusLock.
     */
    container?: ElementType;

    /**
     * Пробрасывает класс для компонента `dialog`.
     */
    dialogClassName?: string;

    /**
     * Пробрасывает класс для компонента `container`.
     */
    containerClassName?: string;

    /**
     * По умолчанию модалка ставит фокус на первый фокусируемый элемент.
     * `disableAutoFocus` отключает это поведение
     */
    disableAutoFocus?: boolean;

    /**
     * Отключает анимацию в модалке
     */
    disableAnimations?: boolean;

    /**
     * Запрещает скролл страницы под модальным окном.
     * Проблема встречается на iOS.
     */
    preventBodyScroll?: boolean;

    /**
     * Раскрывает модалку на весь экран
     */
    fullScreen?: boolean;

    /**
     * Включает мобильную вёрстку
     */
    isMobile?: boolean;

    /**
     * Позиция модалки по вертикали
     * @default 'middle'
     */
    verticalAlign?: 'top' | 'middle' | 'bottom' | 'stretch';

    /**
     * Управляет видимостью модалки
     */
    isVisible: boolean;

    /** Удалять компонент из DOM при скрытии */
    unmountOnHide?: boolean;

    /**
     * Управляет отображением крестика в модалке
     *
     * @default true
     */
    hasCloseButton?: boolean;

    /**
     * Компонент, который будет использоваться в качестве крестика
     */
    renderCloseButton?: (close: () => void) => ReactNode;

    /**
     * Управляет закрытие модалки при клике на фон и на Escape
     *
     * @default true
     */
    autoclosable?: boolean;

    /**
     * Возвращать фокус в позицию до открытия модалки
     *
     * @default true
     */
    returnFocus?: boolean;

    /**
     * Сбрасывает скролл при повторном открытии
     */
    resetScrollOnOpen?: boolean;

    /**
     * Вызывается при клике на крестик, фон и нажатие Escape.
     */
    onClose?(): void;
}

interface IModalState {
    isInitialized: boolean;
    bodyHideScroll: boolean;
    shake: boolean | null;
    contentContainer: HTMLElement | undefined;
    showAt: number | null; // Время (в миллисекундах), когда модальное окно было показано
}

export const MODAL_ANIMATION_DURATION_MS = 200;

const MODAL_SHAKE_DURATION = 820;

class Modal extends Component<IModalProps, IModalState> {
    /**
     * Задаёт необходимые отступы для контента модалки
     */
    static Content = (props: IModalContentProps): ReactElement => (
        <ModalContext.Consumer>
            {({isMobile}): ReactNode => (
                <Box inset={isMobile ? 3 : 8} {...props} />
            )}
        </ModalContext.Consumer>
    );

    // Возвращает текущее время в миллисекундах
    private static getNow(): number {
        return new Date().valueOf();
    }

    private isOuterClickStarted: boolean = false;

    private readonly rootRef: RefObject<HTMLElement> = createRef<HTMLElement>();

    static readonly defaultProps: IModalProps = {
        verticalAlign: 'middle',
        isVisible: false,
        hasCloseButton: true,
        preventBodyScroll: true,
        returnFocus: true,
        autoclosable: true,
        onClose: noop,
        ...prepareQaAttributes('modal'),
    };

    constructor(props: IModalProps) {
        super(props);

        this.state = {
            isInitialized: props.isVisible,
            bodyHideScroll: Boolean(props.isVisible && props.preventBodyScroll),
            shake: null,
            contentContainer: undefined,
            showAt: props.isVisible ? Modal.getNow() : null,
        };
    }

    componentDidMount(): void {
        document.addEventListener('keydown', this.onKeyDown);

        this.setState({});
    }

    componentDidUpdate(prevProps: IModalProps): void {
        this.initializeOnFirstShow();
        this.updateScrollViewOnShow(prevProps);
        this.updateOnPreventBodyScrollChange(prevProps);
        this.updateScrollViewOnHide(prevProps);
        this.updateShowAt(prevProps);
    }

    componentWillUnmount(): void {
        document.removeEventListener('keydown', this.onKeyDown);
    }

    /**
     * Компонент предназначен для установки начального положения
     * фокуса в модалке
     */
    static AutoFocus = AutoFocusInside;

    private initializeOnFirstShow(): void {
        const {isVisible} = this.props;

        if (isVisible && !this.state.isInitialized) {
            this.setState({isInitialized: true});
        }
    }

    // Обновляет время показа модального окна
    private updateShowAt(prevProps: IModalProps): void {
        const {isVisible} = this.props;

        if (!prevProps.isVisible && isVisible) {
            this.setState({showAt: Modal.getNow()});
        }
    }

    private updateOnPreventBodyScrollChange(prevProps: IModalProps): void {
        const {preventBodyScroll} = this.props;

        if (prevProps.preventBodyScroll !== preventBodyScroll) {
            this.updateScrollView();
        }
    }

    /**
     * Обновление блокировки скрола при показе модального окна
     *
     * @param prevProps
     */
    private updateScrollViewOnShow(prevProps: IModalProps): void {
        const {isVisible, resetScrollOnOpen} = this.props;

        if (isVisible && !prevProps.isVisible) {
            this.setState({shake: null});
            this.updateScrollView();

            if (resetScrollOnOpen && this.rootRef.current) {
                scrollTo({top: 0}, this.rootRef.current);
            }
        }
    }

    /**
     * Обновление блокировки скрола при скрывани модального окна
     *
     * @param prevProps
     */
    private updateScrollViewOnHide(prevProps: IModalProps): void {
        const {isVisible, disableAnimations} = this.props;

        if (prevProps.isVisible && !isVisible) {
            if (disableAnimations) {
                this.updateScrollView();
            } else {
                setTimeout(
                    () => this.updateScrollView(),
                    MODAL_ANIMATION_DURATION_MS,
                );
            }
        }
    }

    private updateScrollView(): void {
        const {isVisible, preventBodyScroll} = this.props;

        this.setState({
            bodyHideScroll: Boolean(isVisible && preventBodyScroll),
        });
    }

    private readonly handleCloseButtonClick = (): void => {
        this.close();
    };

    private readonly handleMouseDown = (evt: SyntheticEvent): void => {
        const {isVisible} = this.props;

        if (
            isVisible &&
            !this.state.contentContainer?.contains(evt.target as Node)
        ) {
            this.isOuterClickStarted = true;
        }

        evt.stopPropagation();
    };

    private readonly handleMouseUp = (evt: SyntheticEvent): void => {
        const {isVisible} = this.props;

        if (
            isVisible &&
            this.isOuterClickStarted &&
            !this.state.contentContainer?.contains(evt.target as Node)
        ) {
            this.autoClose();
        }

        this.isOuterClickStarted = false;

        evt.stopPropagation();
    };

    private autoClose(): void {
        const {autoclosable} = this.props;
        const {shake} = this.state;

        if (autoclosable) {
            this.close();
        } else if (!shake) {
            this.shake();
        }
    }

    private shake(): void {
        const {disableAnimations} = this.props;

        if (!disableAnimations) {
            this.setState({shake: true});

            setTimeout(() => {
                this.setState({shake: false});
            }, MODAL_SHAKE_DURATION);
        }
    }

    private readonly onKeyDown = (evt: KeyboardEvent): void => {
        const {isVisible} = this.props;

        if (isVisible && evt.key === 'Escape') {
            evt.preventDefault();
            evt.stopPropagation();
            evt.stopImmediatePropagation();
            this.autoClose();
        }
    };

    private close(): void {
        const {onClose} = this.props;
        const {showAt} = this.state;
        const now = Modal.getNow();

        /**
         * Если с момента открытия модалки не прошло достаточное время для отображения окна,
         * то запрещаем его закрывать. Делаем это во избежании эффекта от двойного щелчка по
         * кнопке-открывашке модалки, когда оно закрывается, не успев открыться.
         */
        if (!showAt || now - showAt < 300) {
            return;
        }

        if (onClose) {
            onClose();
        }
    }

    private setContainerRef = (elem: HTMLElement): void => {
        this.setState({contentContainer: elem});
    };

    private getModalElement(): ReactNode {
        const {
            children,
            container,
            dialogClassName,
            containerClassName,
            className,
            disableAnimations,
            disableAutoFocus,
            fullScreen,
            hasCloseButton,
            isMobile,
            isVisible,
            returnFocus,
            verticalAlign,
            renderCloseButton,
        } = this.props;

        const {bodyHideScroll, shake} = this.state;

        return (
            <LayersContext.Consumer>
                {(layer): ReactNode => {
                    // + 2 для шторки в шторке
                    const newLayer = layer + 2;

                    return (
                        <section
                            className={cx(
                                'root',
                                {
                                    root_visible: isVisible,
                                    root_noAnimation: disableAnimations,
                                    root_fullscreen: fullScreen,
                                    root_mobile: isMobile,
                                },
                                `root_${verticalAlign}`,
                                className,
                            )}
                            style={{
                                zIndex: newLayer * MODAL_LAYER_MULTIPLIER,
                            }}
                            ref={this.rootRef}
                            onMouseDown={this.handleMouseDown}
                            onClick={this.handleMouseUp}
                            {...prepareQaAttributes(this.props)}
                            data-visible={isVisible}
                        >
                            <section
                                className={cx('dialog', dialogClassName, {
                                    dialog_startShake: isVisible && shake,
                                    dialog_stopShake:
                                        shake !== null && isVisible && !shake,
                                })}
                            >
                                <ScopeContext.Provider
                                    value={this.state.contentContainer}
                                >
                                    <LayersContext.Provider value={newLayer}>
                                        <ModalContext.Provider
                                            value={{isMobile}}
                                        >
                                            <FocusLock
                                                className={cx(
                                                    'content',
                                                    containerClassName,
                                                )}
                                                as={
                                                    container ||
                                                    (fullScreen
                                                        ? undefined
                                                        : CardWithForwardRef)
                                                }
                                                lockProps={
                                                    fullScreen
                                                        ? undefined
                                                        : {shadow: 'popup'}
                                                }
                                                autoFocus={!disableAutoFocus}
                                                returnFocus={returnFocus}
                                                disabled={
                                                    !isVisible || isMobile
                                                }
                                                ref={this.setContainerRef}
                                            >
                                                {children}
                                                {renderCloseButton
                                                    ? renderCloseButton(
                                                          this
                                                              .handleCloseButtonClick,
                                                      )
                                                    : hasCloseButton && (
                                                          <button
                                                              className={cx(
                                                                  'close-button',
                                                              )}
                                                              type="button"
                                                              onClick={
                                                                  this
                                                                      .handleCloseButtonClick
                                                              }
                                                              {...prepareQaAttributes(
                                                                  {
                                                                      parent: this
                                                                          .props,
                                                                      current:
                                                                          'closeButton',
                                                                  },
                                                              )}
                                                          >
                                                              <CloseIcon />
                                                          </button>
                                                      )}
                                            </FocusLock>

                                            {bodyHideScroll && (
                                                <HideBodyVerticalScroll />
                                            )}
                                        </ModalContext.Provider>
                                    </LayersContext.Provider>
                                </ScopeContext.Provider>
                            </section>
                        </section>
                    );
                }}
            </LayersContext.Consumer>
        );
    }

    render(): ReactNode {
        const {isInitialized} = this.state;
        const {unmountOnHide, isVisible} = this.props;

        if (
            !isInitialized ||
            typeof document === 'undefined' ||
            (!isVisible && unmountOnHide)
        ) {
            return null;
        }

        return ReactDOM.createPortal(this.getModalElement(), document.body);
    }
}

export interface IModalContentProps extends IBoxProps {}

export default Modal;
