import React, {useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';
import {INavigationItem} from 'types/common/INavigationItem';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18nNew/avia-route';

import Box from 'components/Box/Box';
import Link from 'components/Link/Link';
import Text from 'components/Text/Text';
import Heading from 'components/Heading/Heading';
import Separator from 'components/Separator/Separator';
import Raskrivashka from 'components/Raskrivashka/Raskrivashka';

import cx from './Navigation.scss';

interface INavigationContainerProps extends IWithClassName {
    items: INavigationItem[];
    expandable?: boolean;
    onLinkClick?: (
        e: React.MouseEvent<HTMLElement, MouseEvent> & {
            target: HTMLAnchorElement;
        },
    ) => void;
}

const NavigationContainer: React.FC<INavigationContainerProps> = ({
    className,
    expandable,
    items,
    onLinkClick,
}) => {
    const deviceType = useDeviceType();

    const navigationContent = useMemo(
        () =>
            items.map((item, index) => {
                return (
                    <div key={index}>
                        <Link
                            url={`#${item.anchor}`}
                            theme="ghost"
                            className={cx(
                                'navigationItemLink',
                                deviceMods('navigationItemLink', deviceType),
                            )}
                            onClick={onLinkClick}
                        >
                            {item.text}
                        </Link>
                    </div>
                );
            }),
        [items, onLinkClick, deviceType],
    );

    if (items.length === 0) {
        return null;
    }

    if (expandable) {
        return (
            <Raskrivashka
                className={className}
                label={i18nBlock.navigationTitle()}
                contentIsInvolved={false}
            >
                <Box
                    x="4"
                    below={deviceType.isDesktop ? 3 : undefined}
                    between={deviceType.isDesktop ? 5 : undefined}
                    textSize="m"
                >
                    <Separator />
                    <div
                        className={cx(
                            'expandableNavItemsContainer',
                            deviceMods(
                                'expandableNavItemsContainer',
                                deviceType,
                            ),
                        )}
                    >
                        {navigationContent}
                    </div>
                </Box>
            </Raskrivashka>
        );
    }

    return (
        <div className={className}>
            <Box below={deviceType.isDesktop ? 5 : 4}>
                <Heading level={2}>
                    <Text size="l">{i18nBlock.navigationTitle()}</Text>
                </Heading>
            </Box>

            <Box between={3} textSize="s">
                {navigationContent}
            </Box>
        </div>
    );
};

export default NavigationContainer;
