import React, {useCallback} from 'react';
import {useLocation} from 'react-router-dom';

import {EProjectName} from 'constants/common';
import {URLs} from 'constants/urls';

import {ECommonGoal} from 'utilities/metrika/types/goals/common';
import {IWithClassName} from 'types/withClassName';
import {TMetrikaGoal} from 'utilities/metrika/types/goals/all';

import {reachGoal} from 'utilities/metrika';
import {internalUrl} from 'utilities/url';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getNavigationProjectText from 'utilities/getNavigationProjectText';

import Link from 'components/Link/Link';

import cx from './NavigationContainer.scss';

interface INavigationProps extends IWithClassName, IWithQaAttributes {}

export const NAVIGATION_PROJECTS = [
    EProjectName.AVIA,
    EProjectName.TRAINS,
    EProjectName.HOTELS,
    EProjectName.BUSES,
];

const tabClickMetrikaGoals: PartialRecord<EProjectName, TMetrikaGoal> = {
    [EProjectName.AVIA]: ECommonGoal.AVIA_TAB,
    [EProjectName.TRAINS]: ECommonGoal.TRAINS_TAB,
    [EProjectName.HOTELS]: ECommonGoal.HOTELS_TAB,
    [EProjectName.BUSES]: ECommonGoal.BUS_TAB,
};

const Navigation: React.FC<INavigationProps> = props => {
    const {className} = props;

    const {pathname} = useLocation();
    const deviceType = useDeviceType();

    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    const onNavigateToProject = (...args: any[]) => {};

    const onLinkClick = useCallback(
        (project: EProjectName, isActive: boolean) => {
            const goal = isActive
                ? ECommonGoal.ACTIVE_TAB
                : tabClickMetrikaGoals[project];

            if (goal) {
                reachGoal(goal);
            }

            onNavigateToProject(project);
        },
        [onNavigateToProject],
    );

    return (
        <nav
            {...prepareQaAttributes(props)}
            className={cx(className, 'root', {
                root_mobile: deviceType.isMobile,
            })}
        >
            {NAVIGATION_PROJECTS.map((project): React.ReactNode => {
                const link = URLs[project];

                const isActiveLink = pathname.startsWith(link);

                return (
                    <Link
                        key={project}
                        aria-current={isActiveLink ? 'page' : undefined}
                        className={cx('link', {
                            link_active: isActiveLink,
                        })}
                        theme="none"
                        to={internalUrl(link)}
                        onClick={(): void => onLinkClick(project, isActiveLink)}
                        {...prepareQaAttributes({
                            parent: props,
                            current: project,
                        })}
                    >
                        {getNavigationProjectText(project)}
                    </Link>
                );
            })}
        </nav>
    );
};

export default React.memo(Navigation);
