import React, {useCallback, useRef} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {URLs} from 'constants/urls';

import {IWithClassName} from 'src/types/withClassName';
import {EUserAvatarSize} from 'components/User/types';
import {ETripsGoal} from 'utilities/metrika/types/goals/trips';
import {ECommonGoal} from 'utilities/metrika/types/goals/common';

import userFullInfoSelector from 'selectors/common/userFullInfoSelector';

import {params} from 'utilities/metrika';
import {internalUrl} from 'utilities/url';
import getNavigationProjectText from 'utilities/getNavigationProjectText';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';
import {useReachGoal} from 'utilities/metrika/useReachGoal';

import * as i18nBlock from 'i18nNew/account';
import * as i18nNavBlock from 'i18nNew/common-navigation';

import Flex from 'components/Flex/Flex';
import Link from 'components/Link/Link';
import User from 'components/User/User';
import Separator from 'components/Separator/Separator';
import LinkButton from 'components/LinkButton/LinkButton';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import YandexMessenger from 'components/YandexMessenger/YandexMessenger';
import MobileSideSheet from 'components/MobileSideSheet/MobileSideSheet';
import NotificationDot from 'components/NotificationDot/NotificationDot';
import Label, {ECornersType, ELabelThemeType} from 'components/Label/Label';
import {NAVIGATION_PROJECTS} from 'components/NavigationContainer/NavigationContainer';
import PlusInfo from './components/PlusInfo/PlusInfo';
import ActiveTripsList from './components/ActiveTripsList/ActiveTripsList';
import BurgerMenuIcon from 'icons/24/Burger';
import GiftOutlineIcon from 'icons/16/GiftOutline';

import useElementClick from './hooks/useElementClick';

import cx from './NavigationSideSheet.scss';

export interface INavigationSideSheetProps
    extends IWithClassName,
        IWithQaAttributes {}

const NavigationSideSheet: React.FC<INavigationSideSheetProps> = props => {
    const {className} = props;

    const dispatch = useDispatch();
    const {userInfo, unreadMailCount} = useSelector(userFullInfoSelector);
    const isNewTrip = true;
    const fetchUserUnreadMailCount = useCallback(() => {
        // dispatch(mailActions.fetchUserUnreadMailCount());
    }, [dispatch]);

    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    const onNavigateToProject = (...args: any[]) => {};
    const rootRef = useRef<HTMLDivElement>(null);

    const {value: isVisible, setTrue: show, setFalse: hide} = useBoolean(false);

    const handleRootClick: React.MouseEventHandler<HTMLDivElement> =
        useElementClick({
            elementSelector: 'a[href^="/"]',
            rootRef,
            onClick: hide,
        });

    const handleTripsLinkClick = useReachGoal(
        ETripsGoal.TRIPS_LINK_NAVIGATION_SIDE_SHEET_CLICK,
    );

    const handleCertificateClick = useCallback(() => {
        params({common: {sideSheetCertificateClick: true}});
    }, []);

    return (
        <>
            <LinkButton
                className={className}
                onClick={show}
                theme="black"
                {...prepareQaAttributes({
                    parent: props,
                    current: 'toggleButton',
                })}
            >
                <NotificationDot size="m" dotVisible={isNewTrip}>
                    <BurgerMenuIcon />
                </NotificationDot>
            </LinkButton>
            <MobileSideSheet isVisible={isVisible} onClose={hide}>
                <div
                    className={cx('navigationSideSheet')}
                    ref={rootRef}
                    onClick={handleRootClick}
                    {...prepareQaAttributes(props)}
                >
                    <User
                        className={cx('user')}
                        userInfo={userInfo}
                        avatarSize={EUserAvatarSize.M}
                        unreadMailCount={unreadMailCount}
                        withName
                        fetchUserUnreadMailCount={fetchUserUnreadMailCount}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'user',
                        })}
                    />
                    <Flex tag="section" flexDirection="column" between={6}>
                        {NAVIGATION_PROJECTS.map(
                            (projectName): React.ReactNode => (
                                <Link
                                    className={cx('link')}
                                    theme="black"
                                    key={projectName}
                                    to={internalUrl(URLs[projectName])}
                                    onClick={(): void =>
                                        onNavigateToProject(projectName)
                                    }
                                >
                                    {getNavigationProjectText(projectName)}
                                </Link>
                            ),
                        )}
                        <Link
                            className={cx('link', 'certificateLink')}
                            theme="black"
                            url={internalUrl(URLs.promoCertificatesUrl)}
                            target="_blank"
                            rel="noopener noreferrer"
                            onClick={handleCertificateClick}
                        >
                            <TextWithIcon
                                text={i18nNavBlock.certificates()}
                                iconLeft={GiftOutlineIcon}
                            />
                            <Label
                                theme={ELabelThemeType.ATTENTION}
                                cornersType={ECornersType.ROUNDED}
                            >
                                {i18nNavBlock.newItem()}
                            </Label>
                        </Link>
                    </Flex>
                    <Separator margin={7} />
                    <Flex tag="section" flexDirection="column" between={7}>
                        <Flex flexDirection="column" between={4}>
                            <Link
                                to={internalUrl(URLs.accountTrips)}
                                className={cx('link')}
                                theme="black"
                                onClick={handleTripsLinkClick}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'tripsLink',
                                })}
                            >
                                <Flex alignItems="center">
                                    <span className={cx('tripLinkText')}>
                                        {i18nBlock.menuDotTrips()}
                                    </span>

                                    {isNewTrip && (
                                        <div className={cx('newTripDot')} />
                                    )}
                                </Flex>
                            </Link>
                            {isAuthUser(userInfo) && (
                                <ActiveTripsList
                                    {...prepareQaAttributes({
                                        parent: props,
                                        current: 'activeTripsList',
                                    })}
                                />
                            )}
                        </Flex>

                        {isAuthUser(userInfo) && (
                            <Link
                                to={internalUrl(URLs.accountPassengers)}
                                className={cx('link')}
                                theme="black"
                            >
                                {i18nBlock.menuDotPassengers()}
                            </Link>
                        )}
                    </Flex>
                    <Separator margin={7} />
                    <PlusInfo userInfo={userInfo} />

                    <YandexMessenger
                        className={cx('messenger')}
                        label={i18nBlock.chatWithSupport()}
                        labelSize="m"
                        entrypoint="sideSheet"
                        metrikaGoal={ECommonGoal.PORTAL_CHAT_MENU_CLICK}
                    />
                </div>
            </MobileSideSheet>
        </>
    );
};

export default React.memo(NavigationSideSheet);
