import React from 'react';

import {IWithClassName} from 'types/withClassName';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nCommonFiltersBlock from 'i18nNew/common-filters';
import * as i18nAviaSearchBlock from 'i18nNew/avia-search';

import Box from 'components/Box/Box';
import Button from 'components/Button/Button';
import {Intent} from 'components/Intent/Intent';
import Heading from 'components/Heading/Heading';
import {Center} from 'projects/avia/components/Center/Center';

import cx from './NoResultsAfterFiltration.scss';

export interface IFilterBadgeProps {
    text: string;
    handler: () => void;
}

interface IBaseProps extends IWithClassName {
    title?: string;
    subtitle?: string;
    resetButtonText?: string;
}

interface IResetAllProps extends IBaseProps {
    filters?: IFilterBadgeProps[];
    onReset: () => void;
}

interface IResetByOneProps extends IBaseProps {
    filters: IFilterBadgeProps[];
    onReset?: () => void;
}

type TProps = IResetAllProps | IResetByOneProps;

const NoResultsAfterFiltration: React.FC<TProps> = ({
    className,
    title = i18nAviaSearchBlock.filtersDashNoDashResultsDashTitle(),
    subtitle = i18nAviaSearchBlock.filtersDashNoDashResultsDashSubtitle(),
    filters = [],
    onReset,
    resetButtonText = i18nCommonFiltersBlock.resetDashAll(),
}) => {
    const deviceType = useDeviceType();

    const {isMobile} = deviceType;

    return (
        <Center
            className={cx('root', deviceMods('root', deviceType), className)}
        >
            <Box below={isMobile ? 5 : 8} between={isMobile ? 2 : 3}>
                <Heading level="2">{title}</Heading>

                {subtitle && (
                    <Box textSize="m" textColor="primary">
                        {subtitle}
                    </Box>
                )}
            </Box>

            {Boolean(filters.length) && (
                <Box className={cx('filters')} between="2" inline>
                    {filters.map(({text, handler}, index) => (
                        <Intent
                            className={cx('filterIntent')}
                            size={isMobile ? 'l' : 'm'}
                            key={index}
                            text={text}
                            onCancel={handler}
                        />
                    ))}
                </Box>
            )}

            {Boolean(onReset) && (
                <Button
                    className={cx('resetButton')}
                    theme="attention"
                    width={isMobile ? 'max' : 'auto'}
                    size={isMobile ? 'l' : 'm-inset'}
                    onClick={onReset}
                >
                    {resetButtonText}
                </Button>
            )}
        </Center>
    );
};

export default React.memo(NoResultsAfterFiltration);
