import React, {useMemo} from 'react';

import {EProjectName} from 'constants/common';

import {IWithClassName} from 'types/withClassName';

import {ORDER_STATUS_VALUES} from 'projects/account/lib/orders/statuses';
import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceModMobile} from 'utilities/stylesUtils';
import {getBackLinkInfo} from 'components/OrderHeader/utilities/getBackLinkInfo';

import * as i18nHappyPageBlock from 'i18nNew/happyPage';
import * as i18nBlock from 'i18nNew/account-OrderHeader';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Link from 'components/Link/Link';
import BackArrowIcon from 'icons/16/ArrowBack';
import CheckCircleIcon from 'icons/16/CheckCircle';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import OrderIcon from 'projects/account/components/OrderIcon/OrderIcon';
import OrderStatus from 'projects/account/components/OrderStatus/OrderStatus';

import cx from './OrderHeader.scss';

interface IOrderHeader extends IWithClassName, IWithQaAttributes {
    number?: string;
    type:
        | EProjectName.TRAINS
        | EProjectName.HOTELS
        | EProjectName.AVIA
        | EProjectName.BUSES;
    title?: string;
    status?: ORDER_STATUS_VALUES;
    showSuccessBadge: boolean;
    tripId?: string | null;
}

const OrderHeader: React.FC<IOrderHeader> = props => {
    const {className, title, number, type, status, showSuccessBadge, tripId} =
        props;

    const deviceType = useDeviceType();

    const backLinkInfo = getBackLinkInfo(tripId);

    const successTitle = useMemo(() => {
        if (!showSuccessBadge) {
            return '';
        }

        if (title) {
            return title;
        }

        switch (type) {
            case EProjectName.TRAINS: {
                return i18nHappyPageBlock.trainTitle();
            }

            case EProjectName.HOTELS: {
                return i18nHappyPageBlock.hotelTitle();
            }

            case EProjectName.AVIA: {
                return i18nHappyPageBlock.aviaTitle();
            }

            case EProjectName.BUSES: {
                return i18nHappyPageBlock.busTitle();
            }
        }
    }, [title, type, showSuccessBadge]);

    return (
        <Flex
            className={cx(
                'root',
                deviceModMobile('root', deviceType),
                className,
            )}
            flexDirection={deviceType.isMobile ? 'column' : 'row'}
            alignItems={deviceType.isMobile ? 'flex-start' : 'center'}
            justifyContent="space-between"
            {...prepareQaAttributes(props)}
        >
            {successTitle ? (
                <div
                    className={cx('successPart')}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'successBadge',
                    })}
                >
                    <CheckCircleIcon
                        className={cx('successIcon')}
                        width={24}
                        height={24}
                    />
                    <Text
                        size="xl"
                        weight="bold"
                        color="current"
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'title',
                        })}
                    >
                        {successTitle}
                    </Text>
                </div>
            ) : (
                <Flex className={cx('backPart')} alignItems="center">
                    <Link
                        to={backLinkInfo.url}
                        onClick={backLinkInfo.handleClick}
                    >
                        <TextWithIcon
                            text={backLinkInfo.text}
                            iconLeft={BackArrowIcon}
                            size="m"
                        />
                    </Link>
                </Flex>
            )}
            {Boolean(number) && (
                <Flex
                    className={cx('numberPart')}
                    alignItems="center"
                    inline
                    between={2}
                >
                    <div
                        className={cx('numberBlock')}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'numberBlock',
                        })}
                    >
                        <OrderIcon type={type} className={cx('icon')} />

                        <Text size="m" color="primary">
                            {i18nBlock._number({
                                number: number ?? '',
                            })}
                        </Text>
                    </div>

                    {status && (
                        <OrderStatus
                            status={status}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'status',
                            })}
                        />
                    )}
                </Flex>
            )}
        </Flex>
    );
};

export default React.memo(OrderHeader);
