import React, {useCallback, useMemo} from 'react';

import {deviceMods} from 'utilities/stylesUtils';
import {CHAR_ELLIPSES} from 'utilities/strings/charCodes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {getBodyPages} from 'components/Pagination/utilities/getBodyPages';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Button from 'components/Button/Button';
import ArrowLeftIcon from 'icons/16/ArrowLeft';
import ArrowRightIcon from 'icons/16/ArrowRight';
import LinkButton from 'components/LinkButton/LinkButton';

import cx from './Pagination.scss';

interface IPaginationProps {
    page: number;
    itemsCount: number;
    itemsPerPage: number;
    onPageChange: (page: number) => void;
}

const Pagination: React.FC<IPaginationProps> = props => {
    const deviceType = useDeviceType();
    const {itemsCount, page, itemsPerPage, onPageChange} = props;
    const pagesCount = Math.ceil(itemsCount / itemsPerPage);
    const pages = useMemo(
        () => Array.from({length: pagesCount}, (_item, index) => index + 1),
        [pagesCount],
    );

    const handlePrevClick = useCallback(() => {
        if (page > 1) {
            onPageChange(page - 1);
        }
    }, [page, onPageChange]);

    const handleNextClick = useCallback(() => {
        if (page < pagesCount) {
            onPageChange(page + 1);
        }
    }, [page, pagesCount, onPageChange]);

    const handlePageNumberClick = useCallback(
        event => {
            const {
                target: {
                    dataset: {number},
                },
            } = event;

            if (number) {
                onPageChange(parseInt(number, 10));
            }
        },
        [onPageChange],
    );

    if (pages.length < 2) {
        return null;
    }

    const body = getBodyPages(pages, page, deviceType.isDesktop ? 5 : 3);

    return (
        <Flex
            className={cx('root', deviceMods('root', deviceType))}
            inline
            nowrap
            flexDirection="row"
            between="2"
            alignItems="center"
            justifyContent="center"
        >
            <Button
                className={cx('button', {button_hidden: page === 1})}
                shape="circle"
                icon={<ArrowLeftIcon />}
                onClick={handlePrevClick}
            />

            <Flex
                inline
                flexDirection="row"
                between={deviceType.isDesktop ? 2 : 0}
            >
                <LinkButton
                    className={cx('page', {page_active: page === pages[0]})}
                    data-number={pages[0]}
                    onClick={handlePageNumberClick}
                    theme="none"
                >
                    {pages[0]}
                </LinkButton>

                {body[0] > pages[0] + 1 && (
                    <Text className={cx('separator')} color="secondary">
                        {CHAR_ELLIPSES}
                    </Text>
                )}

                {body.map(item => (
                    <LinkButton
                        className={cx('page', {page_active: item === page})}
                        key={item}
                        data-number={item}
                        onClick={handlePageNumberClick}
                        theme="none"
                    >
                        {item}
                    </LinkButton>
                ))}

                {body[body.length - 1] < pages[pages.length - 1] - 1 && (
                    <Text className={cx('separator')} color="secondary">
                        {CHAR_ELLIPSES}
                    </Text>
                )}

                <LinkButton
                    className={cx('page', {
                        page_active: page === pages[pages.length - 1],
                    })}
                    data-number={pages[pages.length - 1]}
                    onClick={handlePageNumberClick}
                    theme="none"
                >
                    {pages[pages.length - 1]}
                </LinkButton>
            </Flex>

            <Button
                className={cx('button', {button_hidden: page === pagesCount})}
                shape="circle"
                icon={<ArrowRightIcon />}
                onClick={handleNextClick}
            />
        </Flex>
    );
};

export default Pagination;
