import React, {useCallback, useMemo, useState} from 'react';

import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {sendPaymentCounter} from 'utilities/solomon';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {appendQueryParams} from 'utilities/url/appendQueryParams';
import getCanUseApplePay from 'components/PaymentIframe/utilities/getCanUseApplePay';

import useOldTrustMessage from 'components/PaymentIframe/hooks/useOldTrustMessage';
import useTrustSdk from 'components/PaymentIframe/hooks/useTrustSdk';
import useCustomFinishPaymentEvent from 'components/PaymentIframe/hooks/useCustomFinishPaymentEvent';
import SecureIFrameProxy from 'components/SecureIFrameProxy/SecureIFrameProxy';

import cx from './PaymentIframe.scss';

interface IPaymentIframeProps extends IWithClassName, IWithQaAttributes {
    paymentUrl: string;
    iframeProps?: React.IframeHTMLAttributes<HTMLIFrameElement>;
    onFrameLoaded?: () => void;
    onStartPayment?: () => void;
    onFinishPayment?: () => void;
    onPaymentSuccess?: () => void;
    onPaymentError?: (error: string) => void;
    onStart3dsPayment?: () => void;
    onBeforeSubmit?: () => void;
    onLoad?: () => void;
    /**
     * Событие в соломон - загрузка фрейма
     */
    loadEvent: string;
    /**
     * Событие в соломон - успешная оплата
     */
    successEvent: string;
    /**
     * Событие в соломон - ошибка оплаты
     */
    errorEvent: string;
    /**
     * Используется новая форма траста
     * Использовать sdk, вместо старой нашей подписки на события
     * Использовать новые размеры для фрейма
     */
    isNewTrust: boolean;
}

const PaymentIframe: React.FC<IPaymentIframeProps> = props => {
    const {
        paymentUrl,
        iframeProps,
        className,
        onBeforeSubmit,
        onLoad,
        onStart3dsPayment,
        onFinishPayment,
        onPaymentSuccess,
        onPaymentError,
        onStartPayment,
        onFrameLoaded,
        loadEvent,
        successEvent,
        errorEvent,
        isNewTrust,
    } = props;

    const deviceType = useDeviceType();

    const canUseApplePay = getCanUseApplePay();

    const [height, setHeight] = useState<number | string | undefined>(
        undefined,
    );
    const [is3ds, setIs3ds] = useState<boolean>(false);

    const handleFrameLoaded = useCallback((): void => {
        onFrameLoaded?.();

        sendPaymentCounter(loadEvent);
    }, [onFrameLoaded, loadEvent]);

    const handleStart3dsPayment = useCallback((): void => {
        setIs3ds(true);

        onStart3dsPayment?.();
    }, [setIs3ds, onStart3dsPayment]);

    const handleFinishPayment = useCallback((): void => {
        setIs3ds(false);

        onFinishPayment?.();
    }, [setIs3ds, onFinishPayment]);

    const handlePaymentSuccess = useCallback((): void => {
        onPaymentSuccess?.();
        sendPaymentCounter(successEvent);

        handleFinishPayment();
    }, [handleFinishPayment, onPaymentSuccess, successEvent]);

    const handlePaymentError = useCallback(
        (error: string): void => {
            onPaymentError?.(error);
            sendPaymentCounter(errorEvent);

            handleFinishPayment();
        },
        [handleFinishPayment, onPaymentError, errorEvent],
    );

    const handleIframeBodyResize = useCallback(
        (h: number | string | undefined): void => {
            /**
             * В новой форме траста для десктопа используем размер фрейма от траста,
             * для старой формы делаем только для тача
             */
            const canHandleResize = isNewTrust
                ? deviceType.isDesktop
                : deviceType.isMobile;

            if (!canHandleResize || h === undefined) {
                return;
            }

            setHeight(h);
        },
        [isNewTrust, deviceType],
    );

    const paymentUrlWithApplePay = useMemo(() => {
        if (!isNewTrust) {
            return paymentUrl;
        }

        return appendQueryParams(paymentUrl, {
            can_use_apple_pay: canUseApplePay,
        });
    }, [paymentUrl, isNewTrust, canUseApplePay]);

    const trustApiHost = useMemo(() => {
        const url = new URL(paymentUrl);

        return url.origin;
    }, [paymentUrl]);

    useCustomFinishPaymentEvent({
        onFinishPayment: handleFinishPayment,
    });

    useOldTrustMessage({
        enabled: !isNewTrust,
        onIframeBodyResize: handleIframeBodyResize,
        onPaymentError: handlePaymentError,
        onPaymentSuccess: handlePaymentSuccess,
        onStart3dsPayment: handleStart3dsPayment,
        onFrameLoaded: handleFrameLoaded,
        onStartPayment,
        onBeforeSubmit,
    });

    useTrustSdk({
        enabled: isNewTrust,
        canUseApplePay,
        trustApiHost,
        onIframeBodyResize: handleIframeBodyResize,
        onPaymentError: handlePaymentError,
        onPaymentSuccess: handlePaymentSuccess,
        onStart3dsPayment: handleStart3dsPayment,
        onFrameLoaded: handleFrameLoaded,
        onStartPayment,
        onBeforeSubmit,
    });

    return (
        /**
         * Использование SecureIFrameProxy - обязательно.
         * Форма траста может сделать редирект на сайт вне наших политик csp - что приведет к ошибкам.
         * @link https://st.yandex-team.ru/TRAVELFRONT-6511
         */
        <SecureIFrameProxy
            className={cx(className, 'root', deviceMods('root', deviceType), {
                root_3ds: is3ds,
                newTrust: isNewTrust,
            })}
            style={{
                height,
            }}
            src={paymentUrlWithApplePay}
            onLoad={onLoad}
            {...iframeProps}
            {...prepareQaAttributes(props)}
        />
    );
};

export default React.memo(PaymentIframe);
