import {useCallback, useEffect} from 'react';

import {
    ETrust3DSStatus,
    ETrustMessageType,
    isBeforeSubmit,
    isPaymentError,
    isPaymentLoaded,
    isPaymentStarted,
    isPaymentSuccess,
    TRUST_SOURCE,
    TTrustMessage,
} from 'utilities/solomon';

interface IParams {
    enabled: boolean;
    onPaymentSuccess: () => void;
    onPaymentError: (error: string) => void;
    onStart3dsPayment: () => void;
    onIframeBodyResize: (height: number) => void;
    onFrameLoaded?: () => void;
    onStartPayment?: () => void;
    onBeforeSubmit?: () => void;
}

/**
 * Прослушивание событий для старой формы
 */
export default function useOldTrustMessage(params: IParams): void {
    const {
        enabled,
        onStart3dsPayment,
        onFrameLoaded,
        onStartPayment,
        onPaymentError,
        onPaymentSuccess,
        onIframeBodyResize,
        onBeforeSubmit,
    } = params;

    const handleChildrenPostMessage = useCallback(
        (e: MessageEvent): void => {
            const {data} = e;

            /* Messages from Trust */
            if (data?.source === TRUST_SOURCE) {
                const message = data as TTrustMessage;

                if (
                    message.type === ETrustMessageType.SECURE_3D_STATUS &&
                    message.data === ETrust3DSStatus.STARTED
                ) {
                    onStart3dsPayment();
                }

                if (isPaymentLoaded(message)) {
                    onFrameLoaded?.();
                }

                if (isPaymentStarted(message)) {
                    onStartPayment?.();
                }

                if (isPaymentError(message)) {
                    onPaymentError(message.type);
                }

                if (isPaymentSuccess(message)) {
                    onPaymentSuccess();
                }

                if (isBeforeSubmit(message)) {
                    onBeforeSubmit?.();
                }

                if (message.type === ETrustMessageType.RESIZE) {
                    onIframeBodyResize(message.data.height);
                }
            }
        },
        [
            onStart3dsPayment,
            onPaymentError,
            onPaymentSuccess,
            onIframeBodyResize,
            onStartPayment,
            onFrameLoaded,
            onBeforeSubmit,
        ],
    );

    useEffect(() => {
        if (!enabled) {
            return;
        }

        window.addEventListener('message', handleChildrenPostMessage, false);

        return (): void => {
            window.removeEventListener('message', handleChildrenPostMessage);
        };
    }, []);
}
