import {useEffect, useMemo} from 'react';
import TrustSDK from '@yandex-trust-sdk/base';

interface IParams {
    enabled: boolean;
    canUseApplePay: boolean;
    trustApiHost: string;
    onPaymentSuccess: () => void;
    onPaymentError: (error: string) => void;
    onStart3dsPayment: () => void;
    onIframeBodyResize: (height: string | undefined) => void;
    onFrameLoaded?: () => void;
    onStartPayment?: () => void;
    onBeforeSubmit?: () => void;
}

export default function useTrustSdk(params: IParams): void {
    const {
        enabled,
        canUseApplePay,
        trustApiHost,
        onFrameLoaded,
        onStartPayment,
        onStart3dsPayment,
        onPaymentSuccess,
        onPaymentError,
        onIframeBodyResize,
        onBeforeSubmit,
    } = params;
    const trustSdk = useMemo(
        () => TrustSDK.create({apiHost: trustApiHost, frameOrigin: '*'}),
        [trustApiHost],
    );

    useEffect(() => {
        if (!enabled) {
            return;
        }

        return trustSdk.listenPayment(message => {
            switch (message.type) {
                case 'form-ready': {
                    onFrameLoaded?.();

                    return;
                }
                case 'success': {
                    onPaymentSuccess();

                    return;
                }
                case 'error': {
                    onPaymentError(message.reason);

                    return;
                }
                case 'resize': {
                    onIframeBodyResize(message.size.height);

                    return;
                }
                case 'form-submit': {
                    onBeforeSubmit?.();
                    onStartPayment?.();

                    return;
                }
                case 'auth-start': {
                    onStart3dsPayment();

                    return;
                }

                case 'abort':
                case 'auth-end':
                case 'discount':
                    return;
            }
        });
    }, []);

    useEffect(() => {
        if (!enabled || !canUseApplePay) {
            return;
        }

        return trustSdk.applePay();
    }, []);
}
