import React, {useState} from 'react';

import {IWithClassName} from 'types/withClassName';

import {deviceModMobile} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {CHAR_MINUS} from 'utilities/strings/charCodes';
import {CurrencyType} from 'utilities/currency/CurrencyType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlockAviaPlus from 'i18nNew/avia-plusPromo2021';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import YandexPlusSignColorGradientS from 'icons/illustrations/YandexPlusSignColorGradient/YandexPlusSignColorGradientS';
import TriggerDetailsButton from 'components/TriggerDetailsButton/TriggerDetailsButton';
import Heading from 'components/Heading/Heading';
import {EMessageBoxPopupTheme} from 'components/MessageBoxPopup/MessageBoxPopup';
import Link from 'components/Link/Link';
import Price from 'components/Price/Price';

import ScopeContext from 'contexts/ScopeContext';

import cx from './PlusInfo.scss';

interface IPlusInfoProps extends IWithClassName, IWithQaAttributes {
    plusPoints: number;
    popupText: string;
    popupTitle: string;
    isDiscount?: boolean;
    currency?: CurrencyType;
    priceClassName?: string;
}

const PlusInfo: React.FC<IPlusInfoProps> = props => {
    const {
        plusPoints,
        className,
        popupText,
        popupTitle,
        isDiscount,
        currency,
        priceClassName,
    } = props;
    const deviceType = useDeviceType();
    const [ref, setRef] = useState(undefined);

    return (
        <ScopeContext.Provider value={ref || undefined}>
            <Flex
                className={cx('plusInfoContainer', className)}
                justifyContent="space-between"
                innerRef={setRef}
                {...prepareQaAttributes(props)}
            >
                <Text
                    size="s"
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'label',
                    })}
                >
                    {isDiscount
                        ? i18nBlockAviaPlus.createOrderWithdrawText()
                        : i18nBlockAviaPlus.createOrderRefundText()}
                    <TriggerDetailsButton
                        theme={EMessageBoxPopupTheme.WHITE}
                        className={cx('detailsButton')}
                        iconClassName={cx(
                            'detailsButtonIcon',
                            isDiscount && 'detailsButtonIcon_hidden',
                        )}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'triggerDetailsButton',
                        })}
                    >
                        <div
                            className={cx(
                                'popupContainer',
                                deviceModMobile('popupContainer', deviceType),
                            )}
                        >
                            <Heading level="3">{popupTitle}</Heading>
                            <Text
                                tag="div"
                                className={cx('popupContainer_text')}
                            >
                                {popupText}
                            </Text>
                            <Link
                                url={'YANDEX_PLUS_URL'}
                                className={cx('popupContainer_link')}
                            >
                                {i18nBlockAviaPlus.plusPopupMoreInfo()}
                            </Link>
                        </div>
                    </TriggerDetailsButton>
                </Text>
                {isDiscount ? (
                    <Price
                        className={priceClassName}
                        value={plusPoints || 0}
                        color="alert"
                        currency={currency || CurrencyType.RUB}
                        prefix={CHAR_MINUS}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'discountPrice',
                        })}
                    />
                ) : (
                    <TextWithIcon
                        text={plusPoints || 0}
                        size="s"
                        spaceBetween="1"
                        iconLeftProps={{className: cx('plusIcon')}}
                        iconLeft={YandexPlusSignColorGradientS}
                        iconSize={12}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'plusPoints',
                        })}
                    />
                )}
            </Flex>
        </ScopeContext.Provider>
    );
};

export default PlusInfo;
