import React, {useCallback, useEffect, useRef} from 'react';
import {useInView} from 'react-intersection-observer';

import {IWithClassName} from 'types/withClassName';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {
    getIconBySize,
    getIconSize,
} from 'components/PlusInfoPopup/utilities/sizeUtils';
import getLabel from 'components/PlusInfoPopup/utilities/getLabel';

import * as i18nBlock from 'i18nNew/components-PlusInfoPopup';

import {EMessageBoxPopupTheme} from 'components/MessageBoxPopup/MessageBoxPopup';
import Link from 'components/Link/Link';
import LinkButton from 'components/LinkButton/LinkButton';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import MessageBoxPopupOrBottomSheet from 'components/MessageBoxPopupOrBottomSheet/MessageBoxPopupOrBottomSheet';

import cx from './PlusInfoPopup.scss';

const POPUP_POSITIONS = [EPopupDirection.TOP, EPopupDirection.BOTTOM];
const POPUP_ANIMATION_DURATION = 1000;

export type TSize = 's-inset' | 'm';

export type TLabelDescriptionType = 'short' | 'full';

const S_ICON_SIZES = {
    width: 12,
    height: 12,
};

const M_ICON_SIZES = {
    width: 16,
    height: 16,
};

interface IPlusInfoPopupProps extends IWithClassName, IWithQaAttributes {
    iconClassName?: string;
    /** Количество баллов Плюса */
    points?: number;
    eligible?: boolean;
    popupPositions?: EPopupDirection[];

    labelSize?: TSize;
    labelDescriptionType?: TLabelDescriptionType;
}

const PlusInfoPopup: React.FC<IPlusInfoPopupProps> = props => {
    const defaultContent = (
        <>
            <h3 className={cx('header')}>{i18nBlock.header()}</h3>
            <div className={cx('description')}>
                {props.eligible
                    ? i18nBlock.descriptionPlus()
                    : i18nBlock.descriptionV2()}
            </div>
            <Link className={cx('link')} theme="normal" url={'YANDEX_PLUS_URL'}>
                {i18nBlock.moreInfoLink()}
            </Link>
        </>
    );
    const {
        className,
        iconClassName,
        children = defaultContent,
        points: plusPoints,
        popupPositions,
        labelSize = 's-inset',
        labelDescriptionType = 'short',
    } = props;

    const rootRef = useRef(null);
    const {value: isVisible, setTrue: show, setFalse: hide} = useBoolean(false);

    const {
        value: canCheckVisibility,
        setTrue: withCheckVisibility,
        setFalse: withoutCheckVisibility,
    } = useBoolean(false);
    const {ref: contentRef, inView: isContentInView} = useInView({
        rootMargin: '-70px 0px 0px 0px',
        threshold: 0.9,
    });

    useEffect((): void => {
        if (isVisible) {
            setTimeout(withCheckVisibility, POPUP_ANIMATION_DURATION);
        } else {
            setTimeout(withoutCheckVisibility, POPUP_ANIMATION_DURATION);
        }
    }, [isVisible, withCheckVisibility, withoutCheckVisibility]);

    useEffect((): void => {
        if (canCheckVisibility && !isContentInView) {
            hide();
        }
    }, [canCheckVisibility, isContentInView, hide]);

    const handleClickShow = useCallback(
        e => {
            e.stopPropagation();
            show();
        },
        [show],
    );

    const handleClickHide = useCallback(
        (
            e:
                | React.MouseEvent<HTMLButtonElement, MouseEvent>
                | Event
                | undefined,
        ) => {
            e?.stopPropagation?.();

            hide();
        },
        [hide],
    );

    const boundaryRef =
        typeof document === 'undefined' ? undefined : {current: document.body};

    return (
        <>
            <LinkButton
                theme="black"
                innerRef={rootRef}
                className={cx('root', className)}
                onClick={handleClickShow}
                {...prepareQaAttributes(props)}
            >
                <TextWithIcon
                    textClassName={cx('plusLabel')}
                    iconLeft={getIconBySize(labelSize)}
                    iconSize={getIconSize(labelSize)}
                    iconLeftClassName={iconClassName}
                    iconLeftProps={
                        labelSize === 's-inset' ? S_ICON_SIZES : M_ICON_SIZES
                    }
                    text={getLabel(labelDescriptionType, plusPoints)}
                    size={labelSize}
                    spaceBetween={1}
                />
            </LinkButton>
            <MessageBoxPopupOrBottomSheet
                isVisible={isVisible}
                onClose={handleClickHide}
                messageBoxProps={{
                    theme: EMessageBoxPopupTheme.WHITE,
                    direction: popupPositions || POPUP_POSITIONS,
                    anchorRef: rootRef,
                    boundaryRef,
                }}
            >
                <div className={cx('content')} ref={contentRef}>
                    {children}
                </div>
            </MessageBoxPopupOrBottomSheet>
        </>
    );
};

export default PlusInfoPopup;
