import React from 'react';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import {EYandexPlusApplicationMode} from 'types/hotels/offer/IHotelOffer';

import {getUserPlusInfo} from 'selectors/common/userInfoSelector';

import {deviceMods} from 'utilities/stylesUtils';
import IPrice from 'utilities/currency/PriceInterface';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useExperiments} from 'utilities/hooks/useExperiments';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';

import * as i18nBlock from 'i18nNew/components-PlusModeBlock';

import Text from 'components/Text/Text';
import Card from 'components/Card/Card';
import Price from 'components/Price/Price';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import YandexPlusSignBwS from 'icons/illustrations/YandexPlusSignBw/YandexPlusSignBwS';
import AutoHeightAnimateControl from 'components/AutoHeightAnimateControl/AutoHeightAnimateControl';
import PlusModeBlockRadioButton from 'components/PlusModeBlock/components/PlusModeBlockRadioButton';

import cx from './PlusModeBlock.scss';

export interface IPlusModeBlockProps extends IWithClassName {
    plusMode: EYandexPlusApplicationMode;
    onChangePlusMode: (mode: EYandexPlusApplicationMode) => void;

    topupPlusPoints: number;
    withdrawPlusPoints: number | undefined;
    priceAfterPlusWithdraw?: IPrice;

    isLoading?: boolean;
}

const PlusModeBlock: React.FC<IPlusModeBlockProps> = props => {
    const {
        className,
        onChangePlusMode,
        topupPlusPoints,
        withdrawPlusPoints,
        priceAfterPlusWithdraw,
        plusMode,
        isLoading,
    } = props;
    const isWithdrawDisabled = !withdrawPlusPoints;
    const isWithdrawBetterThanTopup =
        withdrawPlusPoints && withdrawPlusPoints >= topupPlusPoints;

    const userPlusInfo = useSelector(getUserPlusInfo);
    const hasPlus = userPlusInfo.hasPlus;

    const deviceType = useDeviceType();
    const {priceAfterPlusWithdrawTop} = useExperiments();

    if (!hasPlus) return null;

    return (
        <div className={cx(deviceMods('root', deviceType), className)}>
            <Text color="plus" size="l" weight="medium">
                {i18nBlock.title()}
            </Text>
            <PlusModeBlockRadioButton
                mode={plusMode}
                isLoading={isLoading}
                onChange={onChangePlusMode}
                topupPlusPoints={topupPlusPoints}
                withdrawPlusPoints={withdrawPlusPoints}
                isWithdrawDisabled={isWithdrawDisabled}
            />
            {priceAfterPlusWithdraw &&
                priceAfterPlusWithdrawTop &&
                isWithdrawBetterThanTopup && (
                    <AutoHeightAnimateControl
                        canShowComponent={
                            plusMode === EYandexPlusApplicationMode.TOPUP
                        }
                        contentClassName={cx('priceAfterPlusDrawContainer')}
                    >
                        <Card
                            className={cx('priceAfterPlusDraw')}
                            radius="s"
                            x="2"
                            y="2"
                        >
                            <TextWithIcon
                                size="s"
                                iconLeft={YandexPlusSignBwS}
                                iconLeftProps={{
                                    width: 16,
                                    height: 16,
                                }}
                                text={insertJSXIntoKey(
                                    i18nBlock.priceAfterPlusDraw,
                                )({
                                    price: (
                                        <Price {...priceAfterPlusWithdraw} />
                                    ),
                                })}
                            />
                        </Card>
                    </AutoHeightAnimateControl>
                )}
        </div>
    );
};

export default PlusModeBlock;
