import React, {useCallback, useEffect, useState} from 'react';

import {EYandexPlusApplicationMode} from 'types/hotels/offer/IHotelOffer';
import {IWithClassName} from 'types/withClassName';

import * as i18nBlock from 'i18nNew/hotels-PlusModeBlockRadioButton';

import cx from 'components/PlusModeBlock/PlusModeBlock.scss';
import RadioButtonSlide from 'components/RadioButtonSlide/RadioButtonSlide';
import PlusModeBlockRadioItem from 'components/PlusModeBlock/components/PlusModeBlockRadioItem/PlusModeBlockRadioItem';

export interface IPlusModeBlockRadioButtonProps extends IWithClassName {
    mode: EYandexPlusApplicationMode;
    isLoading?: boolean;
    onChange(mode: EYandexPlusApplicationMode): void;
    topupPlusPoints: number;
    withdrawPlusPoints?: number;
    isWithdrawDisabled: boolean;
}

const PlusModeBlockRadioButton: React.FC<IPlusModeBlockRadioButtonProps> =
    props => {
        const {
            className,
            mode,
            isLoading,
            onChange,
            topupPlusPoints,
            withdrawPlusPoints,
            isWithdrawDisabled,
        } = props;
        const [value, setValue] = useState<EYandexPlusApplicationMode>(mode);

        useEffect(() => {
            const nextMode = isWithdrawDisabled
                ? EYandexPlusApplicationMode.TOPUP
                : mode;

            if (nextMode !== value) {
                setValue(nextMode);
                onChange(nextMode);
            }
        }, [isWithdrawDisabled, mode, onChange, value]);

        const handleOnChange = useCallback(
            (newMode: EYandexPlusApplicationMode) => {
                onChange(newMode);
                setValue(newMode);
            },
            [onChange],
        );

        return (
            <RadioButtonSlide
                className={cx('radioButton', className)}
                value={value}
                disabled={isLoading}
                onChange={handleOnChange}
            >
                <RadioButtonSlide.Option
                    value={EYandexPlusApplicationMode.TOPUP}
                >
                    <PlusModeBlockRadioItem
                        title={i18nBlock.topupPoints()}
                        plusPoints={topupPlusPoints}
                        isActive={value === EYandexPlusApplicationMode.TOPUP}
                        isLoading={isLoading}
                        disabled={isLoading}
                    />
                </RadioButtonSlide.Option>
                <RadioButtonSlide.Option
                    value={EYandexPlusApplicationMode.WITHDRAW}
                    disabled={isWithdrawDisabled}
                >
                    <PlusModeBlockRadioItem
                        title={i18nBlock.withdrawPoints()}
                        plusPoints={withdrawPlusPoints}
                        isActive={value === EYandexPlusApplicationMode.WITHDRAW}
                        isLoading={isLoading}
                        disabled={isLoading || isWithdrawDisabled}
                    />
                </RadioButtonSlide.Option>
            </RadioButtonSlide>
        );
    };

export default React.memo(PlusModeBlockRadioButton);
