import React, {RefObject, ReactNode} from 'react';
import {compose} from '@bem-react/core';

import {IWithClassName} from 'types/withClassName';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {mergeRefs} from '@yandex-lego/components/lib/mergeRefs';

import {
    Popup as PopupDesktop,
    withThemeNormal,
    withTargetAnchor,
    IPopupTargetAnchorProps,
    withNonvisual,
} from '@yandex-lego/components/Popup/desktop';

import cx from './Popup.scss';

const LegoPopup = compose(
    withThemeNormal,
    withTargetAnchor,
    withNonvisual,
)(PopupDesktop);

LegoPopup.prototype.forwardRefs = function (): void {
    mergeRefs(this.props.targetRef, this.targetRef);
};

export interface IPopupProps extends IPopupTargetAnchorProps, IWithClassName {
    /** Элемент, относительно которого необходимо позиционировать попап. */
    anchor: RefObject<HTMLElement>;

    /** Размер отступа от компонента-триггера */
    gap?: 'small' | 'medium';

    /** Включает/отключает хвостик у попапа */
    hasTail?: boolean;

    /** отображение попапа без тревельной обёртки */
    plain?: boolean;

    nonvisual?: boolean;

    /** Видимость попапа. */
    visible: boolean;

    /**
     * Область в которой необходимо разместить попап
     * @default document.body
     **/
    scope?: RefObject<HTMLElement>;

    children?: React.ReactNode;

    /**
     * Направления раскрытия блока.
     **/
    directions?: EPopupDirection | EPopupDirection[];

    onClose?: () => void;

    targetRef?: RefObject<HTMLElement>;

    zIndex?: number;

    style?: React.CSSProperties;

    addonAfter?: ReactNode;

    /**
     * Отступ от края окна браузера
     * @default 0
     * источник: withTargetAnchor
     */
    viewportOffset?: number;

    /**
     * Должен ли попап занимать всю ширину
     */
    fullWidth?: boolean;

    /**
     * При использовании Popup внутри Spring может возникнуть проблема с отображением
     *
     * Описание проблемы:
     * lego popup использует библиотеку popper.js.org которая для элемента в контейнере Spring добавляет
     * data-popper-escaped (This attribute gets applied when the popper escapes the reference element's boundary (and so it appears detached))
     * см. https://popper.js.org/docs/v2/modifiers/hide/
     * и в этом случае lego popup добавляет класс полностью скрывающий popup
     * https://a.yandex-team.ru/arc_vcs/frontend/packages/lego-components/src/Popup/Popup.css?blame=true&peg=522844351ee255a8dedc11c4dc901142ef4a9d25&rev=r7905965
     * данный флаг отменяет такое поведение
     */
    fixPopupInSpring?: boolean;
}

const Popup: React.FC<IPopupProps> = ({
    anchor,
    children,
    directions,
    scope,
    visible,
    zIndex,
    className,
    style,
    plain,
    gap = 'medium',
    nonvisual = true,
    fullWidth = false,
    fixPopupInSpring = false,
    ...props
}) => {
    return (
        <LegoPopup
            visible={visible}
            theme="normal"
            nonvisual={nonvisual}
            target="anchor"
            scope={scope}
            anchor={anchor}
            direction={directions}
            zIndex={zIndex}
            keepMounted={false}
            className={cx(
                'root',
                {
                    root_fullWidth: fullWidth,
                    'fix-popup-in-spring': fixPopupInSpring,
                },
                plain ? className : undefined,
            )}
            style={plain ? style : undefined}
            {...props}
        >
            {plain ? (
                children
            ) : (
                <div
                    className={cx(
                        'popup',
                        `popup_gap-${gap}`,
                        {'fix-popup-in-spring': fixPopupInSpring},
                        className,
                    )}
                    style={style}
                >
                    {children}
                </div>
            )}
        </LegoPopup>
    );
};

export default Popup;
