import React, {useCallback, useMemo, useState, memo} from 'react';

import {IInputProps} from 'components/Input/types/InputProps';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {formatPrice} from 'utilities/currency';
import IPrice from 'utilities/currency/PriceInterface';

import Input from 'components/Input/Input';
import Text from 'components/Text/Text';

import cx from './PriceInput.scss';

interface IPriceInputProps extends Omit<IInputProps, 'onChange'> {
    beforeText?: string;
    currency: IPrice['currency'];
    onValueChange: (newValue: number) => void;
    postfix?: string;
}

const PriceInput: React.FC<IPriceInputProps> = ({
    value = '0',
    currency,
    onFocus,
    onBlur,
    beforeText,
    onValueChange,
    postfix,
    className,
    ...props
}) => {
    const {
        value: focused,
        setTrue: setFocus,
        setFalse: setBlur,
    } = useBoolean(false);

    const [innerText, setInnerText] = useState(value?.toString() || '');

    const handleFocus = useCallback(
        (e: React.FocusEvent<HTMLInputElement>) => {
            setFocus();

            setInnerText(value?.toString() || '');

            onFocus?.(e);
        },
        [setFocus, onFocus, value],
    );

    const handleBlur = useCallback(
        (e: React.FocusEvent<HTMLInputElement>) => {
            const parsed = parseInt(innerText, 10);

            if (parsed) {
                onValueChange(parsed);
            }

            setBlur();
            onBlur?.(e);
        },
        [setBlur, onBlur, innerText, onValueChange],
    );

    const handleChange = useCallback((_, newValue: string) => {
        setInnerText(newValue);
    }, []);

    const beforeTextNode = useMemo(
        () => (
            <Text className={cx('text', 'text_size_l')} color="secondary">
                {beforeText}&nbsp;
            </Text>
        ),
        [beforeText],
    );

    const priceFormatted = formatPrice({value, currency, postfix});

    return (
        <Input
            {...props}
            onFocus={handleFocus}
            onBlur={handleBlur}
            addonBefore={beforeText && beforeTextNode}
            className={cx('input', 'input_flex', className)}
            size="l"
            onChange={handleChange}
            inputMode={focused ? 'numeric' : undefined}
            value={focused ? innerText : priceFormatted}
        />
    );
};

export default memo(PriceInput);
