import React, {useMemo} from 'react';
import {isString} from 'lodash';

import {formatPrice} from 'utilities/currency';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {PriceConverter} from 'utilities/currency/priceConverter';

import * as i18nBlock from 'i18nNew/common-price';

import {IPriceProps} from 'components/Price/Price';

import cx from './PriceWithoutConvertor.scss';

export interface IPriceWithoutConvertorProps extends IPriceProps {
    priceConverter: PriceConverter;
}

const PriceWithoutConvertor: React.FC<IPriceWithoutConvertorProps> = props => {
    const {
        className,
        color,
        value,
        currency,
        isCurrencyShown = true,
        withConversion = true,
        isFrom,
        isUpTo,
        isRound,
        isRoughly,
        prefix,
        prefixDelimiter,
        postfix,
        priceConverter,
    } = props;
    let priceValue = value;
    let priceCurrency = currency;
    const isPreferredCurrency = priceConverter.isPreferredCurrency(currency);

    if (withConversion && !isPreferredCurrency) {
        const convertedPrice = priceConverter.convertToPreferredCurrency({
            value: isString(value) ? parseFloat(value) : value,
            currency,
        });

        priceValue = convertedPrice.value;
        priceCurrency = convertedPrice.currency;
    }

    let price = formatPrice({
        value: priceValue,
        currency: priceCurrency,
        isCurrencyShown,
        isRound: (withConversion && !isPreferredCurrency) || isRound,
        postfix,
    });

    price = isRoughly ? i18nBlock.roughly({price}) : price;

    const prefixNode = useMemo(() => {
        if (!prefix) {
            return null;
        }

        if (prefixDelimiter) {
            return `${prefix}${prefixDelimiter}`;
        }

        return <span className={cx('prefix')}>{prefix}</span>;
    }, [prefix, prefixDelimiter]);

    if (isFrom) {
        price = i18nBlock._from({price});
    } else if (isUpTo) {
        price = i18nBlock.upTo({price});
    }

    return (
        <span
            className={cx('root', className, color && `color_${color}`)}
            {...prepareQaAttributes({
                ...prepareQaAttributes('price'),
                ...props,
            })}
        >
            {prefixNode}
            {price}
        </span>
    );
};

export default PriceWithoutConvertor;
