import {
    FunctionComponent,
    ReactNode,
    useCallback,
    useEffect,
    useRef,
} from 'react';

import {IWithClassName} from 'types/withClassName';

import {useToggle} from 'utilities/hooks/useToggle';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import {Icon} from 'components/Icon/Icon';
import RotatingArrowIcon from 'components/RotatingArrowIcon/RotatingArrowIcon';
import AutoHeightAnimateControl from 'components/AutoHeightAnimateControl/AutoHeightAnimateControl';

import cx from './Raskrivashka.scss';

interface IRaskrivashkaProps extends IWithQaAttributes, IWithClassName {
    label: ReactNode;
    // сигнализирует что скрытый контент был как-то задействован
    contentIsInvolved: boolean;
    children: ReactNode;
    theme?: 'clear' | 'default';
    initialOpened?: boolean;
    withPrerender?: boolean;
    buttonClassName?: string;
    iconClassName?: string;
    contentClassName?: string;

    onClickButton?(): void; // Клик по области-триггеру открытия/закрытия
}

const Raskrivashka: FunctionComponent<IRaskrivashkaProps> = ({
    label,
    contentIsInvolved,
    children,
    theme = 'default',
    initialOpened = false,
    withPrerender = true,
    buttonClassName,
    className,
    iconClassName,
    contentClassName,
    onClickButton,
    ...rest
}) => {
    const [opened, toggleOpened] = useToggle(initialOpened);
    const [openedWithDelay, toggleWithDelay] = useToggle(initialOpened);
    const contentIsRendered = withPrerender || opened || openedWithDelay;
    const delayTimeout = useRef<number>();

    useEffect(
        () => (): void => {
            window.clearTimeout(delayTimeout.current);
        },
        [],
    );

    const toggle = useCallback(() => {
        toggleOpened();
        delayTimeout.current = window.setTimeout(toggleWithDelay, 500);

        if (onClickButton) {
            onClickButton();
        }
    }, [onClickButton, toggleOpened, toggleWithDelay]);

    return (
        <div className={cx(`root_theme_${theme}`, className)}>
            <button
                className={cx(
                    'button',
                    buttonClassName,
                    contentIsInvolved && !opened && 'button_active',
                )}
                type="button"
                onClick={toggle}
                {...prepareQaAttributes(rest)}
            >
                <span className={cx('label')}>{label}</span>
                <span className={cx('icon', iconClassName)}>
                    <Icon>
                        <RotatingArrowIcon rotated={opened} />
                    </Icon>
                </span>
            </button>
            <AutoHeightAnimateControl canShowComponent={opened}>
                <div className={cx('content', contentClassName)}>
                    {contentIsRendered && children}
                </div>
            </AutoHeightAnimateControl>
        </div>
    );
};

export default Raskrivashka;
