import React from 'react';
import {Link} from 'react-router-dom';
import {useInView} from 'react-intersection-observer';

import {ECommonGoal} from 'utilities/metrika/types/goals/common';

import {formatDateRange} from 'utilities/dateUtils';
import {reachGoal} from 'utilities/metrika';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import cx from './Recipe.scss';

export type TRecipe = 'normal' | 'wide' | 'wider' | 'ultrawide' | 'mobile';

export interface IRecipeProps extends IWithQaAttributes {
    /** Ссылка до картинки без суффикса размера. Суффикс подставляется блоком */
    imageUrl?: string;

    /** Откуда отправка */
    fromTitle?: string | React.ReactElement;

    /** Имя проекта */
    project?: string | React.ReactElement;

    /** Пункт назначения в именительном падеже */
    targetCity?: string | React.ReactElement;

    /** Если нужно отрисовать имя страны. В именительном падеже */
    targetCountry?: string | React.ReactElement;

    /** День отправки в ISO */
    dateFrom?: string;

    /** День прибытия в ISO */
    dateTo?: string;

    /** Цена перелета */
    price?: React.ReactNode;

    /**
     * Тип отображения
     * @default 'normal'
     **/
    type?: TRecipe;

    /** Ссылка при клике по плашке */
    href?: string;

    /** Параметр target ссылки */
    target?: string;

    /** Параметр rel ссылки */
    rel?: string;

    /** Ссылка для Link react-router */
    to?: string;

    isLazy?: boolean;

    /** Обработчик клика по карточке */
    onClick?: React.MouseEventHandler;
}

function getLayoutByType(type: TRecipe): string {
    switch (type) {
        case 'normal':
            return 'layout-center';
        case 'wide':
            return 'layout-spread';
        case 'wider':
            return 'layout-spread';
        case 'ultrawide':
            return 'layout-ultrawide';
        case 'mobile':
            return 'layout-side';
    }
}

function getSizeByType(type: TRecipe): string {
    switch (type) {
        case 'normal':
            return 'content_size_desktop-normal';
        case 'wide':
            return 'content_size_desktop-wide';
        case 'wider':
            return 'content_size_desktop-wider';
        case 'ultrawide':
            return 'content_size_desktop-ultrawide';
        case 'mobile':
            return 'content_size_mobile';
    }
}

const Recipe: React.FC<IRecipeProps> = props => {
    const type = props.type || 'normal';

    const [ref, hasIntersected] = useInView({
        triggerOnce: true,
        skip: !props.isLazy,
    });

    const backgroundImage =
        (!props.isLazy || hasIntersected) && props.imageUrl
            ? `url(${props.imageUrl})`
            : undefined;
    const layout = getLayoutByType(type);
    const size = getSizeByType(type);

    if (props.to && props.href) {
        throw new TypeError('Only one of props "to" or "href" must be passed');
    }

    const content = (
        <React.Fragment>
            <div className={cx('cover')} style={{backgroundImage}} ref={ref} />
            <div className={cx('fade')} />

            <div className={cx('content', layout)}>
                <div className={cx('area-title', 'small')}>
                    <span
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'title',
                        })}
                    >
                        {props.fromTitle}
                    </span>
                    <span
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'project',
                        })}
                    >
                        {props.project}
                    </span>
                </div>

                <div className={cx('area-destination')}>
                    <span
                        className={cx('country')}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'from',
                        })}
                    >
                        {props.targetCountry}
                    </span>
                    <span
                        className={cx('bold')}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'to',
                        })}
                    >
                        {props.targetCity}
                    </span>
                </div>

                <div className={cx('area-price')}>
                    {props.dateFrom && (
                        <span className={cx('common')}>
                            {formatDateRange(props.dateFrom, props.dateTo)}
                        </span>
                    )}
                    <span className={cx('price-wrapper', 'bold')}>
                        {props.price}
                    </span>
                </div>
            </div>
        </React.Fragment>
    );

    const commonProps = {
        className: cx('root', 'link', size),
        onClick: (event: React.MouseEvent<Element, MouseEvent>) => {
            reachGoal(ECommonGoal.RECIPE_LINK);
            if (props.onClick) props.onClick(event);
        },
    };

    if (props.to) {
        return (
            <Link
                to={props.to}
                target={props.target}
                rel={props.rel}
                {...commonProps}
                {...prepareQaAttributes(props)}
            >
                {content}
            </Link>
        );
    }

    if (props.href) {
        return (
            <a
                href={props.href}
                target={props.target}
                rel={props.rel}
                {...commonProps}
                {...prepareQaAttributes(props)}
            >
                {content}
            </a>
        );
    }

    if (props.onClick) {
        return (
            <button
                type="button"
                {...commonProps}
                {...prepareQaAttributes(props)}
            >
                {content}
            </button>
        );
    }

    return <div {...commonProps}>{content}</div>;
};

export default Recipe;
