import React from 'react';
import chunk from 'lodash/chunk';

import {IWithDeviceType} from 'types/withDeviceType';

import Box from 'components/Box/Box';
import {TRecipe} from 'components/Recipe/Recipe';

import cx from './RecipeGrid.scss';

interface RecipeGridProps<T> extends IWithDeviceType {
    /**
     * Данные для отрисовки рецептов
     */
    recipesData: T[];

    /**
     * Метод отрисовки рецепта
     * @argument {T} data - данные переданные в `recipesData`
     * @argument {RecipeType} type - тип отображения рецепта
     */
    renderRecipe: (data: T, type: TRecipe, idx: number) => React.ReactNode;
}

export default function RecipeGrid<T>(props: RecipeGridProps<T>) {
    // eslint-disable-next-line react/display-name
    const recipes = props.recipesData.map((data, idx) => (type: TRecipe) => (
        <div className={cx('grid__cell')} key={idx}>
            {props.renderRecipe(data, type, idx)}
        </div>
    ));

    if (props.deviceType.isMobile) {
        return (
            <div className={cx('root-mobile')}>
                <div className={cx('mobile-scroller')}>
                    <RecipeGridMobile recipes={recipes} />
                </div>
            </div>
        );
    }

    return <RecipeGridDesktop recipes={recipes} />;
}

interface LayoutProps {
    recipes: ((type: TRecipe) => React.ReactNode)[];
    isReversed?: boolean;
}

function RecipeGridDesktop(props: LayoutProps) {
    const Layout = getDesktopLayout(props.recipes.length);

    return (
        <div className={cx('root')}>
            <Layout recipes={props.recipes} isReversed={props.isReversed} />
        </div>
    );
}

function getDesktopLayout(recipesLength: number): React.SFC<LayoutProps> {
    switch (recipesLength) {
        case 5:
            return RecipeGridLayout5;
        case 4:
            return RecipeGridLayout4;
        case 3:
            return RecipeGridLayout3;
        case 2:
            return RecipeGridLayout2;
        case 1:
            return RecipeGridLayout1;
        case 0:
            return () => null;
        default:
            return RecipeGridLayoutMassonry;
    }
}

function RecipeGridMobile(props: LayoutProps) {
    return (
        <Box between="2" inline nowrap>
            {props.recipes.map(renderRecipe => renderRecipe('mobile'))}
        </Box>
    );
}

function renderLayout(
    isReversed: boolean | undefined,
    ...children: React.ReactElement[]
) {
    if (isReversed) {
        children.reverse();
    }

    return React.createElement(
        Box,
        {between: 5, inline: true, nowrap: true},
        ...children,
    );
}

function RecipeGridLayout5(props: LayoutProps) {
    return renderLayout(
        props.isReversed,
        <>{props.recipes[0]('normal')}</>,
        <Box className={cx('grid__cell')} between="5">
            {props.recipes.slice(1, 3).map(recipe => recipe('wide'))}
        </Box>,
        <Box className={cx('grid__cell')} between="5">
            {props.recipes.slice(3, 5).map(recipe => recipe('wide'))}
        </Box>,
    );
}

function RecipeGridLayout4(props: LayoutProps) {
    return renderLayout(
        props.isReversed,
        <>{props.recipes[0]('normal')}</>,
        <>{props.recipes[1]('normal')}</>,
        <Box className={cx('grid__cell')} between="5">
            {props.recipes.slice(2, 4).map(recipe => recipe('wide'))}
        </Box>,
    );
}

function RecipeGridLayout3(props: LayoutProps) {
    return renderLayout(
        props.isReversed,
        <>{props.recipes[0]('wide')}</>,
        <>{props.recipes[1]('wide')}</>,
        <>{props.recipes[2]('wide')}</>,
    );
}

function RecipeGridLayout2(props: LayoutProps) {
    return renderLayout(
        props.isReversed,
        <>{props.recipes[0]('wider')}</>,
        <>{props.recipes[1]('wider')}</>,
    );
}

function RecipeGridLayout1(props: LayoutProps) {
    return <>{props.recipes[0]('ultrawide')}</>;
}

function RecipeGridLayoutMassonry(props: LayoutProps) {
    const chunks = chunk(props.recipes, 5);

    return (
        <Box between="5">
            {chunks.map((chunk, idx) => {
                const rowIsReversed = idx % 2 === 1;
                const recipes = rowIsReversed ? chunk.slice().reverse() : chunk;

                return (
                    <RecipeGridDesktop
                        key={idx}
                        recipes={recipes}
                        isReversed={rowIsReversed}
                    />
                );
            })}
        </Box>
    );
}
