import _pickBy from 'lodash/pickBy';
import _identity from 'lodash/identity';

import {IHotelInfo} from 'reducers/depreacted/hotels/hotelPage/hotelInfo/types';

import {getYandexMapsUrl} from 'utilities/maps/yandexMaps';
import {hotelsURLs} from 'projects/depreacted/hotels/utilities/urls';

import {IBaseMarkup, baseMarkup} from './baseMarkup';

function filterSchemaEmptyProperties<
    T extends IAggregateRating | IHotelSchemaMarkupPartial,
>(schema: T): Partial<T> {
    return _pickBy(schema, _identity);
}

export interface IAggregateRating
    extends Partial<{
        '@type': string;
        bestRating: string;
        ratingValue: number;
        reviewCount: number;
    }> {}

export interface IHotelSchemaMarkupPartial {
    name: string;
    image: string;
    url: string | undefined;
    priceRange: string | undefined;
    hasMap: string;
    address: string;
    aggregateRating?: IAggregateRating;
}

export interface IHotelSchemaMarkup
    extends IBaseMarkup,
        Partial<IHotelSchemaMarkupPartial> {
    '@type': 'Hotel';
}

/**
 * https://schema.org/Hotel
 */
export function getHotelSchemaMarkup(
    hotelInfo: IHotelInfo,
): IHotelSchemaMarkup {
    const {
        hotel,
        seoInfo: {schemaOrg},
    } = hotelInfo;
    const {name, priceRange, ratingValue, reviewCount, image, address} =
        schemaOrg;

    const aggregateRating =
        ratingValue || reviewCount
            ? filterSchemaEmptyProperties<IAggregateRating>({
                  '@type': 'AggregateRating',
                  bestRating: '5',
                  ratingValue,
                  reviewCount,
              })
            : undefined;

    const url = hotelsURLs.getAbsoluteHotelUrl({slug: hotel.hotelSlug});
    const {lon, lat} = hotel.coordinates;
    const mapUrl = getYandexMapsUrl(lon, lat);

    return {
        '@type': 'Hotel',
        ...baseMarkup,
        ...filterSchemaEmptyProperties<IHotelSchemaMarkupPartial>({
            name,
            image,
            url,
            priceRange,
            aggregateRating,
            hasMap: mapUrl,
            address,
        }),
    };
}
