import React, {PureComponent} from 'react';
import _noop from 'lodash/noop';

import {IWithClassName} from 'types/withClassName';

import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import cx from './SearchButtonGroup.scss';

type TItem<ItemValue> = {
    value: ItemValue;
    label: string;
};

interface ISearchButtonGroupProps<ItemValue>
    extends IWithClassName,
        IWithQaAttributes {
    buttonClassName: string;
    items: TItem<ItemValue>[];
    activeElementValue: ItemValue;
    activeElementIndex: number;
    onChange: (item: TItem<ItemValue>) => void;
}

class SearchButtonGroup<
    ItemValue extends number | string,
> extends PureComponent<ISearchButtonGroupProps<ItemValue>> {
    static readonly defaultProps = {
        className: '',
        buttonClassName: '',
        activeElementIndex: 0,
        onChange: _noop,
        items: [],
    };

    private handleClick = (index: number): void => {
        const {onChange, items} = this.props;
        const selectedItem = items[index];

        onChange(selectedItem);
    };

    private isButtonActive(index: number): boolean {
        const {items, activeElementValue, activeElementIndex} = this.props;

        if (activeElementValue) {
            const elementIndex = items.findIndex(
                ({value: elementValue}) => elementValue === activeElementValue,
            );

            return elementIndex === index;
        }

        return activeElementIndex === index;
    }

    private renderButton = (
        {value, label}: TItem<ItemValue>,
        index: number,
    ): React.ReactNode => {
        const {buttonClassName} = this.props;
        const isActive = this.isButtonActive(index);

        return (
            <div
                className={cx(
                    'button',
                    {
                        button_active_yes: isActive,
                    },
                    buttonClassName,
                )}
                onClick={this.handleClick.bind(this, index)}
                key={value}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: String(value),
                })}
            >
                {label}
            </div>
        );
    };

    render(): React.ReactNode {
        const {items, className} = this.props;

        return (
            <div className={cx('searchButtonGroup', className)}>
                {items.map(this.renderButton)}
            </div>
        );
    }
}

export default SearchButtonGroup;
