import React from 'react';

import {
    ESearchFormSize,
    ESearchFormTriggerViewType,
} from 'components/SearchForm/types';
import {IWithClassName} from 'types/withClassName';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import SearchSubmitButton from 'components/SearchSubmitButton/SearchSubmitButton';
import Button, {TButtonTheme, TButtonSize} from 'components/Button/Button';

const MOBILE_BUTTONS_SIZE_MAP: Pick<
    Record<ESearchFormSize, TButtonSize>,
    ESearchFormSize.S | ESearchFormSize.M
> = {
    [ESearchFormSize.S]: 'm',
    [ESearchFormSize.M]: 'l',
};

interface ISubmitButtonProps extends IWithClassName, IWithQaAttributes {
    size: ESearchFormSize;
    triggerViewType: ESearchFormTriggerViewType;
    text: string | undefined;
    isFormExpanded: boolean;
    theme: TButtonTheme | undefined;
}

const SubmitButton: React.FC<ISubmitButtonProps> = props => {
    const {className, size, triggerViewType, text, isFormExpanded, theme} =
        props;

    const {isMobile} = useDeviceType();

    if (size === ESearchFormSize.XL) {
        const isSimple =
            isMobile || triggerViewType === ESearchFormTriggerViewType.TILE;

        return (
            <SearchSubmitButton
                className={className}
                isSimple={isSimple}
                label={text}
                tabIndex={isFormExpanded ? 0 : -1}
                {...prepareQaAttributes(props)}
            />
        );
    }

    return (
        <Button
            className={className}
            theme={theme}
            size={isMobile ? MOBILE_BUTTONS_SIZE_MAP[size] : size}
            type="submit"
            {...prepareQaAttributes(props)}
        >
            {text}
        </Button>
    );
};

export default React.memo(SubmitButton);
