import React, {useEffect, useMemo} from 'react';
import {useInView} from 'react-intersection-observer';

import {MAX_DAYS_RANGE} from 'constants/hotels';

import {EYtpReferer} from 'types/common/IAttribution';
import {IWithClassName} from 'types/withClassName';
import {IRequiredOfferParams} from 'types/hotels/offer/IHotelOffer';

import {parseDate} from 'utilities/dateUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {isV3Response} from 'utilities/crossSaleHotels/isV3Response';
import getFormattedDatesForPeriod from 'projects/avia/utilities/getFormattedDatesForPeriod';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {ROBOT} from 'utilities/dateUtils/formats';
import {getCrossSaleHotelsSearchUrl} from 'projects/avia/utilities/getCrossSaleHotelsSearchUrl';

import * as aviaCrossSaleKeyset from 'i18nNew/avia-crossSale';

import useHotelsCrossSale from 'hooks/useHotelsCrossSale/useHotelsCrossSale';

import Price from 'components/Price/Price';
import CrossSaleMap from 'components/CrossSaleMap/CrossSaleMap';
import CrossSaleMapSkeleton from 'components/SearchHotelsCrossSaleMap/components/CrossSaleMapSkeleton/CrossSaleMapSkeleton';

export interface ICrossSaleRequestParams {
    pointKey: string;
    totalHotelLimit: number;
    checkinDate: string;
    checkoutDate?: string;
    adults?: number;
    childrenAges?: number[];
}

interface ISearchHotelsCrossSaleMapProps
    extends IWithClassName,
        IWithQaAttributes {
    requestParams: ICrossSaleRequestParams;
    ytpReferer: EYtpReferer;
    onCrossSaleShow?(): void;
    onCrossSaleClick?(): void;
}

const SearchHotelsCrossSaleMap: React.FC<ISearchHotelsCrossSaleMapProps> =
    props => {
        const {
            className,
            requestParams,
            ytpReferer,
            onCrossSaleShow,
            onCrossSaleClick,
        } = props;

        const deviceType = useDeviceType();
        const {isDesktop} = deviceType;

        const {fetch, reset, isError, isLoading, data} = useHotelsCrossSale();

        const [ref, wasInView] = useInView({
            threshold: 0.8,
            triggerOnce: true,
        });

        const checkinDateMoment = useMemo(() => {
            return parseDate(requestParams.checkinDate);
        }, [requestParams.checkinDate]);
        const checkoutDateMoment = useMemo(() => {
            return requestParams.checkoutDate
                ? parseDate(requestParams.checkoutDate)
                : checkinDateMoment.clone().add(1, 'day');
        }, [checkinDateMoment, requestParams.checkoutDate]);

        const offerSearchParams: Partial<IRequiredOfferParams> = useMemo(() => {
            return {
                checkinDate: checkinDateMoment.format(ROBOT),
                checkoutDate: checkoutDateMoment.format(ROBOT),
                adults: requestParams.adults,
                childrenAges: requestParams.childrenAges,
            };
        }, [
            checkinDateMoment,
            checkoutDateMoment,
            requestParams.adults,
            requestParams.childrenAges,
        ]);

        const crossSale = useMemo(() => {
            if (isError || isLoading) {
                return null;
            }

            return data && (isV3Response(data) ? data.crossSale : data);
        }, [data, isLoading, isError]);

        const additionalTitle = useMemo(() => {
            if (!crossSale) {
                return;
            }

            const {minPriceInRegion} = crossSale;
            const [checkinDateText, checkoutDateText] =
                getFormattedDatesForPeriod(
                    checkinDateMoment,
                    checkoutDateMoment,
                );

            if (isDesktop) {
                return aviaCrossSaleKeyset.dates({
                    checkinDate: checkinDateText,
                    checkoutDate: checkoutDateText,
                });
            }

            return insertJSXIntoKey(aviaCrossSaleKeyset.datesWithMinPrice)({
                checkinDate: checkinDateText,
                checkoutDate: checkoutDateText,
                priceNode: <Price {...minPriceInRegion} isFrom />,
            });
        }, [checkinDateMoment, checkoutDateMoment, crossSale, isDesktop]);

        const hasCrossSale = Boolean(crossSale);

        const correctCrossSaleParams =
            !checkinDateMoment.isSame(checkoutDateMoment) &&
            checkoutDateMoment.diff(checkinDateMoment, 'days') <=
                MAX_DAYS_RANGE;

        useEffect(() => {
            if (hasCrossSale) {
                onCrossSaleShow?.();
            }
        }, [hasCrossSale, onCrossSaleShow]);

        useEffect(() => {
            if (!correctCrossSaleParams) {
                return;
            }

            fetch(requestParams);

            return (): void => {
                reset();
            };
        }, [fetch, requestParams, reset, correctCrossSaleParams]);

        if (!correctCrossSaleParams || isError) {
            return null;
        }

        if (!crossSale || !wasInView || isLoading) {
            return (
                <CrossSaleMapSkeleton
                    ref={ref}
                    deviceType={deviceType}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'skeleton',
                    })}
                />
            );
        }

        return (
            <CrossSaleMap
                ref={ref}
                className={className}
                crossSaleData={crossSale}
                linkUrl={getCrossSaleHotelsSearchUrl({
                    geoId: crossSale.region.geoId,
                    offerSearchParams,
                    ytpReferer,
                })}
                additionalTitle={additionalTitle}
                onLinkClick={onCrossSaleClick}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'card',
                })}
            />
        );
    };

export default SearchHotelsCrossSaleMap;
