import React from 'react';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18nNew/trains-search-information';

import SearchIcon16 from 'icons/16/Search';
import SearchIcon24 from 'icons/24/Search';

import cx from './SearchInformation.scss';

const ROOT_QA = 'searchInformation';

export interface ISearchInformationProps
    extends IWithClassName,
        IWithDeviceType {
    onClick: () => void;
    from: string;
    to: string;
    when: string;
    passengers?: string;
    withHeading?: boolean;
    classType?: string;
}

const SearchInformation: React.FC<ISearchInformationProps> = props => {
    const {
        className,
        deviceType,
        from,
        to,
        when,
        passengers,
        withHeading = false,
        classType,
        onClick,
    } = props;
    const passengersAndClass = [passengers, classType]
        .filter(Boolean)
        .join(', ');

    const DirectionTag = withHeading ? 'h1' : 'div';

    return (
        <div
            className={cx(
                'root',
                {
                    root_mobile: deviceType.isMobile,
                    root_desktop: deviceType.isDesktop,
                },
                className,
            )}
            onClick={onClick}
            {...prepareQaAttributes(ROOT_QA)}
        >
            <div className={cx('content')}>
                <DirectionTag className={cx('direction')}>
                    <span
                        className={cx('text')}
                        {...prepareQaAttributes({
                            parent: ROOT_QA,
                            current: 'direction',
                        })}
                    >
                        {i18nBlock.direction({
                            from,
                            to,
                        })}
                    </span>
                </DirectionTag>

                <div className={cx('additional')}>
                    <span
                        className={cx('when')}
                        {...prepareQaAttributes({
                            parent: ROOT_QA,
                            current: 'when',
                        })}
                    >
                        {when}
                    </span>
                    <span
                        className={cx('passengersAndClass')}
                        {...prepareQaAttributes({
                            parent: ROOT_QA,
                            current: 'passengersAndClass',
                        })}
                    >
                        {passengersAndClass}
                    </span>
                </div>
            </div>

            <div
                className={cx('iconWrapper')}
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: 'lupa',
                })}
            >
                {deviceType.isDesktop ? (
                    <SearchIcon16 className={cx('icon', 'icon_desktop')} />
                ) : (
                    <SearchIcon24 className={cx('icon', 'icon_mobile')} />
                )}
            </div>
        </div>
    );
};

export default SearchInformation;
