import React, {RefAttributes, useCallback, useMemo} from 'react';

import * as Keys from 'constants/eventKeys';

import {IWithClassName} from 'types/withClassName';

import {IDropdownSwitcherParams} from 'components/Dropdown/Dropdown';

export interface IRenderButtonOptions {
    onClick(): void;
    onKeyDown(event: React.KeyboardEvent): void;
    onFocus?(): void;
    onBlur?(): void;
}

export interface ISwitcherOwnProps
    extends IWithClassName,
        RefAttributes<HTMLLabelElement> {
    labelRef: React.MutableRefObject<HTMLLabelElement | null>;

    renderButton(options: IRenderButtonOptions): React.ReactNode;
    hoverUpper(): void;
    hoverLower(): void;
    selectHovered(): void;
    onFocus?(): void;
    onBlur?(): void;
}

type TSwitcherProps = IDropdownSwitcherParams<ISwitcherOwnProps>;

const Switcher: React.FC<TSwitcherProps> = props => {
    const {
        meta: {visible, hidePopup, showPopup},
        input: {ref},
        componentProps: {
            className,
            labelRef,
            renderButton,
            hoverUpper,
            hoverLower,
            selectHovered,
            onFocus,
            onBlur,
        },
    } = props;

    const labelRefCallback = useCallback(
        (node: HTMLLabelElement | null) => {
            ref(node);

            labelRef.current = node;
        },
        [labelRef, ref],
    );

    const handleKeyDown = useCallback(
        (event: React.KeyboardEvent): void => {
            switch (event.key) {
                case Keys.ARROW_UP: {
                    event.preventDefault();

                    if (visible) {
                        hoverUpper();
                    } else {
                        showPopup();
                    }

                    break;
                }

                case Keys.ARROW_DOWN: {
                    event.preventDefault();

                    if (visible) {
                        hoverLower();
                    } else {
                        showPopup();
                    }

                    break;
                }

                case Keys.ENTER: {
                    if (visible) {
                        event.preventDefault();
                        selectHovered();
                        hidePopup();
                    }

                    break;
                }

                case Keys.ESCAPE: {
                    if (visible) {
                        event.preventDefault();
                        hidePopup();
                    }

                    break;
                }
            }
        },
        [hidePopup, hoverLower, hoverUpper, selectHovered, showPopup, visible],
    );

    const buttonNode = useMemo(() => {
        return renderButton({
            onClick: visible ? hidePopup : showPopup,
            onKeyDown: handleKeyDown,
            onFocus,
            onBlur,
        });
    }, [
        handleKeyDown,
        hidePopup,
        onBlur,
        onFocus,
        renderButton,
        showPopup,
        visible,
    ]);

    return (
        <label className={className} ref={labelRefCallback}>
            {buttonNode}
        </label>
    );
};

export default React.memo(Switcher);
