import React, {
    Component,
    ReactNode,
    ChangeEvent,
    ComponentType,
    createElement,
} from 'react';
import _noop from 'lodash/noop';

import {IWithClassName} from 'types/withClassName';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import SelectButton from './components/SelectButton/SelectButton';

import cx from './SelectOld.scss';

export interface ISelectOption {
    value: string | number;
    text: string;
    disabled?: boolean;
    selected?: boolean;
}

interface ISelectProps extends IWithClassName {
    type: string;
    items: ISelectOption[];
    nativeSelectProps: React.DetailedHTMLProps<
        React.SelectHTMLAttributes<HTMLSelectElement>,
        HTMLSelectElement
    >;
    switcherComponent: ComponentType<any>;
    onChange: (params: {value: string | number; type: string}) => void;
}

class Select extends Component<ISelectProps> {
    static defaultProps: ISelectProps = {
        type: '',
        items: [],
        onChange: _noop,
        nativeSelectProps: {},
        switcherComponent: SelectButton,
    };

    static SelectButton = SelectButton;

    private handleChangeNativeSelect = (
        e: ChangeEvent<HTMLSelectElement>,
    ): void => {
        const {onChange, type} = this.props;
        const selectedValue = e.target.value;

        onChange({value: selectedValue, type});
    };

    private renderNativeOption = ({
        value,
        text,
        disabled,
        selected,
    }: ISelectOption): ReactNode => {
        return (
            <option
                key={value}
                value={value}
                disabled={disabled}
                selected={selected}
                {...prepareQaAttributes(`option_${value}`)}
            >
                {text}
            </option>
        );
    };

    render(): ReactNode {
        const {className, nativeSelectProps, switcherComponent, items} =
            this.props;

        return (
            <div className={cx('select', className)}>
                <div
                    className={cx('nativeSelect')}
                    {...prepareQaAttributes('native-select')}
                >
                    {createElement(switcherComponent)}
                    <select
                        className={cx('nativeSelectButton')}
                        onChange={this.handleChangeNativeSelect}
                        {...nativeSelectProps}
                    >
                        {items.map(this.renderNativeOption)}
                    </select>
                </div>
            </div>
        );
    }
}

export default Select;
