import React from 'react';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {SocialSharingNets, socialNetToData} from './utilities/socialNets';

import cx from './ShareLink.scss';

interface IShareLink extends IWithQaAttributes {
    type: SocialSharingNets;
    url: string;
    directionStyle: 'vertical' | 'horizontal';
    textAfter?: string;
    buttonClassName?: string;
    onClick?: (text?: string) => void;
}

export class ShareLink extends React.PureComponent<IShareLink> {
    private getUrlToSocialNet = (): string => {
        const {hrefStart} = socialNetToData[this.props.type];
        const queryUrl = encodeURIComponent(this.props.url);

        return `${hrefStart}${queryUrl}`;
    };

    private renderIcon = (): React.ReactNode => {
        const {iconComponent} = socialNetToData[this.props.type];

        return (
            <div className={cx('iconWrapper')}>
                {React.createElement(iconComponent)}
            </div>
        );
    };

    private renderTextAfter = (): React.ReactNode => {
        const {textAfter} = this.props;

        if (!textAfter) {
            return null;
        }

        return <div className={cx('textAfter')}>{textAfter}</div>;
    };

    private openInNewWindow = (): void => {
        const {width, height} = socialNetToData[this.props.type].size;
        const href = this.getUrlToSocialNet();

        window.open(
            href,
            '',
            [
                'scrollbars=yes',
                'resizable=yes',
                'menubar=no',
                'toolbar=no',
                'status=no',
                `left=${(screen.width - width) / 2}`,
                `top=${(screen.height - height) / 2}`,
                `width=${width}`,
                `height=${height}`,
            ].join(','),
        );
    };

    private onClick = (): void => {
        const {onClick: onClickFromProps} = this.props;

        onClickFromProps?.(this.props.textAfter);
        this.openInNewWindow();
    };

    render(): React.ReactNode {
        const {directionStyle, buttonClassName} = this.props;

        return (
            <button
                className={cx('button', directionStyle, buttonClassName)}
                type="button"
                onClick={this.onClick}
                {...prepareQaAttributes(this.props)}
            >
                {this.renderIcon()}
                {this.renderTextAfter()}
            </button>
        );
    }
}
